#!/usr/bin/env python

#############################################################################
## 
#  \file mosfet.py
#  \brief MOSFET calculation script for power applications
#  \version 0.2
#  \author Dimitri Denk
#  \date 26.07.2015
#
# This file implements the MOSFET calculation algorithm.
#
# Last modification: 11.10.2016
# 
# Copyright (c) 2015-2016
# Dimitri Denk
#
# Permission to use, copy, modify, distribute and sell this software
# and its documentation for any purpose is hereby granted without fee,
# provided that the above copyright notice appear in all copies and
# that both that copyright notice and this permission notice appear
# in supporting documentation. Author makes no representations about 
# the suitability of this software for any purpose. 
# It is provided "as is" without express or implied warranty.
#
#############################################################################

import numpy as np
import matplotlib.pyplot as plt
from matplotlib.figure import SubplotParams

from mosfets import *
from gatedrivers import *

# Visualization of results
SavePlots = False
ShowPlots = True

# Calculation and visualization limits
Nmax = 10		# Number of MOSFETs
Pmax = 10		# W
Igmax = 25		# Gate current, A
Cgmax = 100		# nF
Pgmax = 0.5		# Gate power losses, W
Tmax = 250		# nsec
Cdrvmax = 25	# uF

# System parameters
fsw = 20e3		# Switching frequency, Hz
Vpwr = 48.		# System voltage, V
Iload = 30.		# Load current, A
Vdrv = 12.		# Gate driver voltage, V

Don = 0.9		#

Pllim = 0.95	# Power losses limit (compared to optimum)
Vdrvrip = 1.0	# Ripple voltage on gate

# MOSFET driver
driver = TC4422
#driver = UCC27322
#driver = NCP81074A
#driver = IdealDriver

# MOSFETs
mosfets = [IRFP4468PbF, IRFB4110PbF, IRFB4410ZPbF]
#mosfets = [SUM70040E, SUM70060E, SUM70090E]
#mosfets = [IRFP4468PbF, IRFP4568PbF, IRFB4110PbF, IRFB4410ZPbF, SUM70040E, SUM70040M, SUP70040E, SUM70060E, SUM70090E, IRF540S, IRF2907ZS, IRLB4030PbF]

# Function for calculation of MOSFET losses
def calc(mosfet, driver, Vpwr, Iload, Vdrv, fsw, N = 1, Don = 0.5):
	# Source current
	Is = Iload / N

	# MOSFET manufacturers specify QG(SW) on their data sheets.
	# For those that don't, it can be approximated by:
	Qgsw = mosfet["Qgd"] + mosfet["Qgs"] / 2.

	# Calculated gate capacitance
	Cg = mosfet["Qg"] / Vdrv
	
	# Maximal possible gate current
	Idrvmax = driver["Imax"] / N
	# Limitation of maximal gate current current by internal gate resistance
	Igmax = Vdrv / mosfet["Rg"]
	# Desired current
	Igon = Qgsw / mosfet["Ton"]
	Igoff = Qgsw / mosfet["Toff"]
	
	# Minimal possible resistance
	Ronmin = Vdrv / Idrvmax
	Roffmin = Vdrv / Idrvmax
	# Calculation of gate resistors
	Ron = max((Vdrv - max(mosfet["Vgs"])) / Igon, mosfet["Rg"] +driver["Roh"], Ronmin)
	Roff = max((min(mosfet["Vgs"]) - 0) / Igoff, mosfet["Rg"] +driver["Rol"], Roffmin)
#	Ron = Roff = mosfet["Rg"]

	# Calculation of optimal current value
	Idrvon = min(Idrvmax, Igmax, Vdrv / Ron)
	Idrvoff = min(Idrvmax, Igmax, Vdrv / Roff)

	# Calculation of driver rise / fall times from load parameters
	Tdrvon = 3 * Ron * Cg
	Tdrvoff = 3 * Roff * Cg
	
	# Limitation of switching time by driver characteristics
	# Percentage of switching charge in total gate charge (from linear approach)
	Tswdrvon = Tdrvon * Qgsw / mosfet["Qg"]
	Tswdrvoff = Tdrvoff * Qgsw / mosfet["Qg"]
	
	# Driver output
	Ton = max([Qgsw / Idrvon, mosfet["Ton"], Tswdrvon])
	Toff = max([Qgsw / Idrvoff, mosfet["Toff"], Tswdrvoff])

	# Calculation of power path capacitor
	Cdrv = 2 * mosfet["Qg"] * Vdrv / Vdrvrip**2
	
	# Calculation of Losses
	Pcond = Don * Is**2 * mosfet["Rds"]
	Pch = fsw * (mosfet["Coss"] + mosfet["Crss"]) * Vpwr**2
	Psw = fsw * (Ton + Toff) / 2 * Vpwr * Is
	Pgate = fsw * mosfet["Qg"] * Vdrv
	# Reverce recovery power on schottky diode
	Pqrr = mosfet["Qrr"] * Vdrv * fsw
	# All losses
	Pmosfet = Pgate + Pcond + Psw + Pch + Pqrr

	return { 
		"Pmosfet" : Pmosfet * N, "Pcond" : Pcond * N, "Psw" : (Psw + Pch + Pqrr) * N,
		"Ton" : Ton, "Toff" : Toff, "Pgate" : Pgate * N, "Idrvon" : Idrvon * N, "Idrvoff" : Idrvoff * N, "Cg" : Cg * N, "Cdrv" : Cdrv * N,
		"Igon" : Igon * N, "Igoff" : Igoff * N, "Rgon" : Ron - (mosfet["Rg"] + driver["Roh"]), "Rgoff" : Roff - (mosfet["Rg"] + driver["Rol"]) }

# Creating of radar plot
gridlines = 5
titles = ["Plos, W", "N", "Igon, A", "Igoff, A", "Cg, nF", "Cdrv, uF", "Toff, nsec", "Ton, nsec", "Pgate, W"]
limits = [Pmax, Nmax, Igmax, Igmax, Cgmax, Cdrvmax, Tmax, Tmax, Pgmax]

fig1 = plt.figure(frameon=False, figsize=(5., 5.))

angles = np.arange(90, 90 + 360, 360.0 / len(titles)) % 360
axes = [fig1.add_axes([0.1, 0.11, 0.8, 0.8], polar=True, label="axes" + str(i)) for i in range(len(titles))]

l, text = axes[0].set_thetagrids(angles, labels = titles, fontsize=12, color="black")
for txt, angle in zip(text, angles):
	if angle > 0 and angle <= 180:
		txt.set_rotation(angle - 90)
	else:
		txt.set_rotation(angle + 90)

for ax in axes[1:]:
	ax.patch.set_visible(False)
	ax.grid("off")
	ax.xaxis.set_visible(False)
	pass

for ax, angle, limit in zip(axes, angles, limits):
	label = []
	for l in np.linspace(limit / gridlines, limit, gridlines):
		if l == round(l):
			label.append(int(l))
		else:
			label.append(l)
	label[0] = ""
		
	l, text = ax.set_rgrids(np.linspace(1. / gridlines, 1., gridlines), angle=angle, labels=label, fontsize=10, color="gray")
	ax.spines["polar"].set_visible(False)
	ax.set_ylim(0., 1.)
	ax.xaxis.grid(True,color='gray',linestyle='--')
	
axes[0].xaxis.set_visible(True)
axes[0].set_xlabel(driver["Name"] + " driver @ " + str(Iload) + "A and " + str(fsw / 1000) + "kHz")
axes[0].yaxis.grid(True,color='lightgray',linestyle='-')

# Calculate all MOSFETs
for mosfet in mosfets:

	result = calc(mosfet, driver, Vpwr, Iload, Vdrv, fsw, 1, Don)
#	print "Pmosfet", result["Pmosfet"], "(Pcond:" + str(round(result["Pcond"], 3)) + ", Psw:" + str(round(result["Psw"], 3)) + ", Pgate:" + str(round(result["Pgate"], 3)) + ")"
	
	nt = range(1, Nmax + 1)
	Pcond = []
	Psw = []
	Pgate = []
	Ptotal = []
	Igon = []
	Igoff = []
	Cgate = []
	Idrvon = []
	Idrvoff = []
	for i in nt:
		result = calc(mosfet, driver, Vpwr, Iload, Vdrv, fsw, i, Don)
		Ptotal.append(result["Pmosfet"])
		Pcond.append(result["Pcond"])
		Psw.append(result["Psw"])
		Pgate.append(result["Pgate"])
		Cgate.append(result["Cg"])
		Idrvon.append(result["Idrvon"])
		Idrvoff.append(result["Idrvoff"])
		Igon.append(result["Igon"])
		Igon.append(result["Igoff"])
		
	# Find optimal number of MOSFETs with minimal power losses
	Pmin = min(Ptotal)
	xlim = optN = Ptotal.index(Pmin)

	# Minimize number of MOSFETs
	for i in xrange(xlim, 0, -1):
		if ((Pmin / Ptotal[i]) >= Pllim):
			optN = i
	optN = nt[optN]

	# Plot of calculation results
	sp = SubplotParams(left=0.1, bottom=0.1, right=0.9, top=0.92, wspace=0., hspace=0.)
	fig = plt.figure(frameon=False, subplotpars=sp, figsize=(7., 5.))
	ax1 = fig.add_subplot(1, 1, 1)

	# Plot of power losses
	ax1.set_title("Power losses of " + mosfet["Name"] + " @" + str(Iload) + "A and " + str(fsw / 1000) + "kHz", fontsize=13)
	ax1.grid(True, color='lightgray', linestyle='-')
	ax1.set_ylim(0., Pmax)
	ax1.set_xlabel("Number of MOSFETs")
	ax1.set_ylabel("Power losses, W")

	ax1.plot(nt, Ptotal, 'k', linewidth=1.5, label='Total', )
	ax1.plot(nt, Pcond, 'g', linewidth=1.5, label='Conductive')
	ax1.plot(nt, Psw, 'r', linewidth=1.5, label='Switching')
	ax1.plot(nt, Pgate, 'b', linewidth=1.5, label='Gate')
	ax1.axvspan(1., nt[xlim], facecolor='g', alpha=0.1)
	ax1.axvline(optN, color='k', linewidth=2.0, linestyle='-', alpha=0.4)
	legend = ax1.legend(bbox_to_anchor=(.05, 0.95), loc=2, borderaxespad=0., fontsize=12)

	# Plot of gate current
	ax2 = ax1.twinx()
	ax2.set_xlim(1, Nmax)
	ax2.set_ylim(0., Igmax)
	ax2.set_ylabel('Gate current, A')

	ax2.plot(nt, Idrvon, 'c--', linewidth=1.5, alpha=0.8, label='Ion', )
	ax2.plot(nt, Idrvoff, 'm--', linewidth=1.5, alpha=0.8, label='Ioff', )
	ax2.plot(nt, [driver["Imax"], ] * len(nt), 'k--', linewidth=1.5, alpha=0.8, label='Idrv', )
	ax2.legend(bbox_to_anchor=(.75, 0.95), loc=2, borderaxespad=0., fontsize=12)

	if SavePlots:
		fig.savefig("./" + mosfet["Name"] + ".png", format='png', dpi=75)

	# Radar diagram values
	N = optN
	result = calc(mosfet, driver, Vpwr, Iload, Vdrv, fsw, N, Don)
	print mosfet["Name"], N, result

	values = [
		result["Pmosfet"] * 1. / Pmax, N * 1. / Nmax, result["Idrvon"] / Igmax, result["Idrvoff"] / Igmax,
		result["Cg"] * 1e9 * 1. / Cgmax, result["Cdrv"] * 1e6 / Cdrvmax,
		result["Toff"] * 1e9 / Tmax, result["Ton"] * 1e9 / Tmax,
		result["Pgate"] * 1. / Pgmax
	]
	axes[0].fill(np.deg2rad(np.r_[angles, angles[0]]), np.r_[values, values[0]], "-", lw=2, alpha = 0.15)
	axes[0].plot(np.deg2rad(np.r_[angles, angles[0]]), np.r_[values, values[0]], "-", lw=2, alpha = 0.4, label = mosfet["Name"])
	
axes[0].legend(bbox_to_anchor=(.65, 1.05), loc=2, borderaxespad=0., fontsize=12)

if SavePlots:
	fig1.savefig("./" + driver["Name"] + ".png", format='png', dpi=75)

if ShowPlots:
	plt.show()
