#!/usr/bin/python
##############################################################################
## 
#  \file dti.py
#  \brief DTI main script and interface functions.
#  \version 0.5
#  \author Dimitri Denk
#  \date 12.08.2011
#
#  This file provides opd calculation functions
#
#  Copyright (c) 2010-2011 
#  DENKTECH
#  http://www.denktech.de
#
#  Permission to use, copy, modify, distribute and sell this software
#  and its documentation for any purpose is hereby granted without fee,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear
#  in supporting documentation. DENKTECH makes no representations about 
#  the suitability of this software for any purpose. 
#  It is provided "as is" without express or implied warranty.
#
#############################################################################

##
# \addtogroup DTI_SCRIPTS scripts
# \ingroup DTI_TOOLS
#
# Python scripts for DTI interferometers
#
# @{

from numpy import *
import time
import sys
import getopt

# import DENKTECH libraries
import dtmath

# import DENKTECH scripts
from iapack import *

# Display command line options
def usage():
	print """
dti <command> [options...]

  command          frame|calibrate|phase|phasedev|opd

  common options
    -h, --help     print this message
    --device       device name
    --save-prefix  save file prefix
    --save-data    save data (npy and pgm file formats)
    --save-error   save error (text file format)
    --save-all     save all intermediate results
    -n             repeat command
    -v             verbose output

  commsnd-dependent options
    -a             make data averaging
    --pzt-max      normalized maximal PZT voltage in range from 0. to 1.
    -pzt-ref       load normalized calibrated PZT voltages from file
                   (use .npy file extension)
    --mask         load mask from file (use .npy file extension)
    --zernike      calculate Zernike polynomial coefficients

  capture frame
    command        frame
    options        -a, <common options>

  calibrate PZT
    command        calibrate
    options        --pzt-max, <common options>
	
  capture frames and calculate phase map
    command        phase
    options        --pzt-ref, <common options>
	 
  get phase map from device (DTI build-in algorithm)
    command        phasedev
    options        --pzt-ref, <common options>

  calculate opd (surface reconstruction)
    command        opd
    options        -a, --pzt-ref, --mask, --zernike, <common options>

  generate report from saved opd
    command        report
    options        none
	
examples
  python dti.py frame --save-data
  python dti.py calibrate --pzt-max=0.85 -a5 --save-data
  python dti.py phase --pzt-ref=test_calibrate_0.npy --save-data
  python dti.py phasedev --pzt-ref=test_calibrate_0.npy --save-data
  python dti.py opd --pzt-ref=test_calibrate_0.npy -n4 --save-data --save-error
  python dti.py report opd.npy opd.pdf
 """

# Main
if __name__ == "__main__":

	print sys.argv[0] + " (version 0.5)"

	#default parameters
	device = "dtidev"
	verb = False
	sve = False
	ref = [ 0., 0.21214471, 0.37421732, 0.53773306, 0.70740915]
	cmd = -1
	mask = None
	ref_fname = None
	mask_fname = None
	save_prefix = "test"
	save_data = False
	save_error = False
	zn = 0
	avr = 1
	rep = 1
	pztmax = 0.85
	points = 8
	shift = 0
	errmon = True

	# parse command line options
	try:
		opts, args = getopt.gnu_getopt(sys.argv[1:], "hva:n:", ["help", "device=", "pzt-ref=", "mask=", "zernike=", "save-data", "save-all", "save-prefix=", "save-error", "pzt-max="])
	except getopt.error, msg:
		print msg
		print "for help use -h"
		sys.exit(1)

	# process options
	for o, a in opts:
		if o in ("-h", "--help"):
			usage()
			sys.exit(0)
		if o in ("--device",):
			device = a
		if o in ("-n",):
			rep = int(a)
		if o in ("--pzt-ref",):
			ref_fname = a
		if o in ("--mask",):
			mask_fname = a
		if o in ("--save-prefix",):
			save_prefix = a
		if o in ("--save-data",):
			save_data = True
		if o in ("--save-error",):
			save_error = True
		if o in ("-a",):
			avr = int(a)
		if o in ("--zernike",):
			zn = int(a)
		if o in ("--pzt-max",):
			pztmax = float(a)
		if o in ("-v",):
			verb = True
		if o in ("--save-all",):
			sve = True
			save_data = True
			save_error = True

	# process arguments
	argc = len(args)
	if argc > 0:
		cmd = args[0]
		if cmd == "frame":
			ref_fname = None
			zn = 0
		# calibrate
		elif cmd == "calibrate":
			ref_fname = None
			zn = 0
		# get phase
		elif cmd == "phase":
			avr = 1
			errmon = False
			zn = 0
		# get phase (device build-in algorithm)
		elif cmd == "phasedev":
			avr = 1
			errmon = False
			zn = 0
		# compare two phase algorithms
		elif cmd == "phasecmp":
			avr = 1
			errmon = False
			zn = 0
		# calculate opd
		elif cmd == "opd":
			pass
		# make report
		elif cmd == "report":
			# process arguments
			if argc > 2:
				fname = args[1]
				opd = load(fname)
				mask = dtmath.make_mask(opd[0], .5)
				opd = opd[1]
				reports.make_pdf_report(args[2], [mask, opd])
			sys.exit(0)
		else:
			print "unsupported command \"" + args[0] + "\""
			sys.exit(3)
	else:
		print "command is not specified"
		print "for help use -h"
		sys.exit(2)

	# load referense in numpy format
	if ref_fname != None:
		ref = load(ref_fname)

	# load mask in numpy format
	if mask_fname != None:
		mask = load(mask_fname)

	# create device
	dev = __import__(device)

	# make pass
	datalist = []
	for t in xrange(rep):
		print "pass " + str(t + 1) + " of " + str(rep)

		if cmd == "calibrate":
			result, data = calibrate(dev, pztmax, points, avr, verb, sve)
			print "calibrated PZT values: " + str(data)
			if result != True:
				print "calibration error"
				sys.exit(4)
			avrdata = array(data, dtype=double)
		else:
			avrdatalist = []
			for i in xrange(avr):
				if verb == False:
					sys.stdout.write(".")
					sys.stdout.flush()
				else:
					print "iteration " + str(i + 1) + " of " + str(avr)
		
				if cmd == "frame":
					data = phase.get_frame(dev, shift, verb)
				elif cmd == "phase":
					data = phase.get_phase(dev, ref, verb, sve)
				elif cmd == "phasedev":
					data = phase.get_phase_dev(dev, ref, verb, sve)
				elif cmd == "phasecmp":
					if verb:
						print "compare phase calculation algorithms"
					data = zeros(1)
				# calculate opd
				elif cmd == "opd":
					phasemap = phase.get_phase_dev(dev, ref, verb, sve)
#					phasemap = phase.get_phase(dev, ref, verb, sve)
					# check mask
					if mask == None:
						mask = dtmath.roundmask(.8, phasemap.shape[1], phasemap.shape[0])
					# calculate opd
					opd = phase.unwrap_phase(mask, phasemap)
					data = opd
	
				#store result
				avrdatalist.append(data.astype(double))
						
				# averaging
				if i == 0:
					avrdata = data.astype(double)
				else:
					avrdata += data.astype(double)

			if verb == False:
				print ""
			
			# calculate results 
			avrdata /= avr
	
			# calculate deviation
			if avr > 1:
				for data in avrdatalist:
					e = data - avrdata
					e = e * e;
					rmsval = pow(e.sum(), .5) / e.size
					print "rms deviation: " + str(rmsval)
			#		if rms > 1e-3:
			#			print "OPD error too big: " + str(rms)

		# append result
		datalist.append(avrdata)

		# calculate zernike coefficients
		if zn > 0:
			zk = zernike.zernike_coefficients(mask, avrdata, zn)
			# output results
			print "zernike coefficients: " + str(zk)

	# calculate deviation
	if errmon:
		errlist = []
		for data in datalist:
			e = data - datalist[0]
			e = e * e;
			rmsval = pow(e.sum(), .5) / e.size
			print "rms error: " + str(rmsval)
			errlist.append(rmsval)
	
	# save results
	if save_data:
		for i in xrange(rep):
			if cmd == "calibrate":
				dat_name = save_prefix + "_" + cmd + "_" + str(i) + ".dat"
				print "saving results into " + dat_name
				utils.save_txt(dat_name, datalist[i])
				npy_name = save_prefix + "_" + cmd + "_" + str(i) + ".npy"
				print "saving results into " + npy_name
				save(npy_name, datalist[i])
			else:
				pgm_name = save_prefix + "_" + cmd + "_" + str(i) + ".pgm"
				npy_name = save_prefix + "_" + cmd + "_" + str(i) + ".npy"
				print "saving results into " + pgm_name
				print "saving results into " + npy_name
				if cmd == "frame":
					dtmath.save_pgm(pgm_name, datalist[i].astype(uint16));
					save(npy_name, datalist[i].astype(uint16))
				elif cmd in ("phase", "phasedev"):
					k = 2 * pi
					dtmath.save_pgm(pgm_name, datalist[i], 1. / pi);
					save(npy_name, datalist[i].astype(uint16))
				elif cmd == "opd":
					dtmath.save_pgm(pgm_name, datalist[i], .1);
					save(npy_name, [mask, datalist[i]])
				else:
					dtmath.save_pgm(pgm_name, datalist[i], .1);
					save(npy_name, datalist[i])
	
	# save error
	if save_error:
		if errmon:
			dat_name = save_prefix + "_" + cmd + "_err.dat"
			print "saving results into " + dat_name
			save(dat_name, errlist)
	
	sys.exit(0)

##
# @}

# EOF

