#! /usr/bin/python
##############################################################################
## 
#  \file calibrate.py
#  \brief PZT calibration.
#  \version 0.3
#  \author Dimitri Denk
#  \date 12.08.2011
#
#  This file implements PTZ calibration function for DTI interferometers
#
#  Copyright (c) 2010-2011
#  DENKTECH
#  www.denktech.de
#
#  Permission to use, copy, modify, distribute and sell this software
#  and its documentation for any purpose is hereby granted without fee,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear
#  in supporting documentation. DENKTECH makes no representations about 
#  the suitability of this software for any purpose. 
#  It is provided "as is" without express or implied warranty.
#
#############################################################################

##
# \addtogroup DTI_SCRIPTS scripts
# \ingroup DTI_TOOLS
#
# Python scripts for DTI interferometers
#
# @{

from numpy import *
import time
import sys
import getopt
# import DENKTECH scripts
import utils

## Calibrate PZT
#
#  This function implements PZT calibration.
#  FFT is used as phase detection algorithm.
#  The calibration refers to vertical interference fringes of flat surface.
#
#  \param port - dtcam object
#  \param umax - normalized maximal PZT voltage, float in range from 0. to 1.
#  \param p - number of acquired points, positive integer
#  \param n - number of measurements, positive integer
#  \param verb - verbose output, boolean
#  \param sve - save intermediate results, boolean
#  \retval res - result of operation, True if success
#  \retval list - normalized calibrated PZT voltages, list of floats in range from 0. to 1.
def calibrate(port, umax, p, n, verb = False, sve = False):
	if verb:
		print "PZT calibration"
		print "PZT max value", umax
		print "number of points", p
		print "number of measurements", n

	# prepare phase shift values
	shift = []
	for i in xrange(p):
		shift.append(float(i) / (p - 1) * umax)
	for i in xrange(p):
		shift.append(shift[p - 1 - i])
	sn = len(shift)
	if verb:
		print "PZT shift values:", shift

	xp = arange(0, dtype=float)
	yp = arange(0, dtype=float)
	
	start = time.time();
	
	print "data acquisition..."
	# make pass
	for t in xrange(n):
		print "pass " + str(t + 1) + " of " + str(n)
		
		# acquire image
		f = []
		for s in shift:
			if verb != True:
				sys.stdout.write(".")
				sys.stdout.flush()
			f.append(port.get_frame(s, verb))
		if verb != True:
			print ""

		# process images
		fp = []
		for i in xrange(sn):
			h, w = f[i].shape
			# make 1D array
#			s = zeros(w, dtype=uint32)
			s = zeros(w, dtype=float32)
			for j in xrange(h):
				s = s + f[i][j]
			s = s / j
			if sve:
				utils.save_txt("signal" + str(i) + ".dat", s)

			# calculate polynomial fit
			sx = arange(len(s), dtype=float32)
			sc = poly1d(polyfit(sx, s, 5))
			s = s - sc(sx)
			if sve:
				utils.save("signalfit" + str(i) + ".dat", s)

			#calculate phase
			sp = fft.fft(s)
			sp = sp / sp.size
			sp = sp[1 : w / 2]
			if sve:
				utils.save("fft" + str(i) + ".dat", abs(sp))
			index = argmax(abs(sp))
			phase = angle(sp[index])
			if verb:
				print "frame: " + str(i) + ", fft index: " + str(index)  + ", phase: " + str(phase)
			fp.append(phase)
		
		#unwrap phase
		fp = array(fp)
		fp -= fp[0]		#phase offset correction
		p0 = fp[0]
		for i in xrange(sn):
			while fp[i] - p0 < -pi / 2:
				fp[i:] = fp[i:] + pi
			while fp[i] - p0 > pi / 2:
				fp[i:] = fp[i:] - pi
			p0 = fp[i]

		# add results
		xp = hstack((xp, shift))
		yp = hstack((yp, fp))

	if sve:
		utils.save("phase.dat", yp)

	print "data processing..."
	# calculate polynomial fit
	z = polyfit(xp, yp, 5)
	y = poly1d(z)

	# correct offset
	yp = yp - y[0]
	if sve:
		utils.savexy("response.dat", xp, yp)
	
	# make curve
	if y[1] < 0:
		y = -y
	y[0] = 0
	fit = zeros(sn / 2)
	for i in xrange(sn / 2):
		fit[i] = y(shift[i])
	
	if sve:
		utils.savexy("fit.dat", shift[0: sn / 2], fit)
	
	# find roots
	root = []
	for y0 in [0, pi / 2, pi, 3 * pi / 2, 2 * pi]:
		y[0] = -y0
		pr = roots(y)
#		print "y0", y0, "roots", pr
	
		# filter roots
		rn = len(pr)
		lr = None
		for i in xrange(rn):
			if(pr[i].imag == 0):
				r = pr[i].real
				if(r >= 0) and (r <= 1.0):
					lr = r
		root.append(lr)
		
	time1 = time.time()

	print "done. time " + str(int((time1 - start) * 1000 + .5)) + "ms"

	# check results
	res = True
	for r in root:
		if r == None:
			res = False

	if verb:
		print "result: " + str(res) + ", values: " + str(root)

	# normalized calibrated PZT voltages
	return res, root

# Display command line options
def usage():
	print """
calibrate [-h] [-d <device>] [-o <file>] [-v] [-s] <max> <p> <n> <r>

    -h    print this message
    -d    device name
    -o    output results in file (use .npy file extension)
    -v    verbose output
    -s    save intermediate results

    <max> normalized maximal PZT voltage
    <p>   number of acquired points
    <n>   number of measurements
    <r>   number of repeats
 """

# Main
if __name__ == "__main__":
	#default parameters
	device = "dti"
	umax = .85
	p = 8
	n = 2
	r = 1
	verb = False
	sve = False
	fname = None

	# parse command line options
	try:
		opts, args = getopt.gnu_getopt(sys.argv[1:], "hd:o:vs")
#		print "opts", opts, "args", args
	except getopt.error, msg:
		print msg
		print "for help use -h"
		sys.exit(2)

	# process options
	for o, a in opts:
		if o in ("-h"):
			usage()
			sys.exit(0)
		if o in ("-d"):
			device = a
		if o in ("-o"):
			fname = a
		if o in ("-s"):
			sve = True
		if o in ("-v"):
			verb = True

	# process arguments
	argc = len(args)
	if argc > 0:
		umax = float(args[0])
	if argc > 1:
		p = int(args[1])
	if argc > 2:
		n = int(args[2])
	if argc > 3:
		r = int(args[3])

	# imdev device
	dev = __import__(device + 'dev')

	# process calibration
	if verb:
		print "parameters:", device + 'dev', verb, sve, umax, p, n, r

	avr = zeros(5)	
	for i in xrange(r):
		res, ref = calibrate(dev, umax, p, n, verb, sve)
		if res != True:
			print "Calibration error"
			sys.exit(3)
		avr = avr + ref
	avr /= r

	# save results in numpy format
	if fname != None:
		print "saving results into " + fname + " ..."
		save(fname, avr)

	# output results
	print "normalized calibrated PZT voltages:"
	for v in avr:
		print v

	sys.exit(0)

##
# @}

# EOF
