##############################################################################
## 
#  \file phase.py
#  \brief DTI phase map processing.
#  \version 0.2
#  \author Dimitri Denk
#  \date 12.04.2011
#
#  This file provides functions to get phase map from DTI interferometers
#
#  Copyright (c) 2010-2011
#  DENKTECH
#  www.denktech.de
#
#  Permission to use, copy, modify, distribute and sell this software
#  and its documentation for any purpose is hereby granted without fee,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear
#  in supdeving documentation. DENKTECH makes no representations about 
#  the suitability of this software for any purpose. 
#  It is provided "as is" without express or implied warranty.
#
#############################################################################

##
# \addtogroup DTI_SCRIPTS scripts
# \ingroup DTI_TOOLS
#
# Python scripts for DTI interferometers
#
# @{

from numpy import *
from math import pi
import time
import sys
import getopt
# imdev DENKTECH libraries
import dtmath

## Get frame with phase shifting
#  
#  This function captures one frame with desired phase shifting
#
#  \param dev - dtcam object
#  \param shift - normalized PZT voltage, float in range from 0. to 1.
#  \param verb - verbose output, boolean
#  \retval frame - image, numpy uint16 array
def get_frame(dev, shift, verb = False):

	start = time.time();

	# get frame
	f = dev.get_frame(shift)

	time1 = time.time()
	if verb:
		print "acquisition time (ms):", int((time1 - start) * 1000)

	return f

## Calculate phase map from N frames
#  
#  This function calculates phase map
#
#  \param img - phase shifted images
#  \param avr - average frames
#  \param verb - verbose output, boolean
#  \param sve - save intermediate results, boolean
#  \retval phase - phase map, numpy float array
def calc_phase(img, avr = False, verb = False):

	start = time.time();

	# average frames
	if avr:
		rn = len(img) / 2
		I = []
		for i in xrange(rn):
			I.append(uint16((uint32(img[i]) + uint32(img[rn * 2 - i - 1])) / 2))
		# calculate phase map
		p = dtmath.phase(I)
			
	else:
		# calculate phase map
		p = dtmath.phase(img)
	
	time1 = time.time()

	if verb:
		print "phase map calculation time (ms):", int((time1 - start) * 1000)
	
	return p
	
## Get phase using N frames
#  
#  This function captures frames and calculates phase map
#
#  \param dev - dtcam device
#  \param ref - reference points for phase shifting, list of floats in range from 0. to 1.
#  \param verb - verbose output, boolean
#  \param sve - save intermediate results, boolean
#  \retval phase - phase map, numpy float array
def get_phase(dev, ref, verb = False, sve = False):

	start = time.time();

	rn = len(ref)

	if verb:
		print "get phase (" + str(rn) + " frames)"

	# acquire frames
	F = []
	# forward steps
	for i in xrange(rn):
		f = get_frame(dev, ref[i], verb)
		F.append(f)
	# back steps
	for i in xrange(rn):
		f = get_frame(dev, ref[rn - i - 1], verb)
		F.append(f)

	time1 = time.time()

	# average frames
	I = []
	for i in xrange(rn):
		I.append(uint16((uint32(F[i]) + uint32(F[rn * 2 - i - 1])) / 2))

	# calculate phase map
	p = dtmath.phase(I)

	if sve:
		if verb:
			print "saving results ..."
		for i in xrange(len(F)):
			dtmath.save_pgm('frame' + str(i) + '.pgm', F[i]);
		for i in xrange(len(I)):
			dtmath.save_pgm('img' + str(i) + '.pgm', I[i]);
		dtmath.save_pgm('phase.pgm', p, 1. / pi);

	time2 = time.time()

	if verb:
		print "acquisition time (ms):", int((time1 - start) * 1000)
		print "processing time (ms):", int((time2 - time1) * 1000)
	
	return p

## Get phase map from device
#  
#  This function gets phase map using device build-in phase calculation algorithm
#
#  \param dev - dtcam device
#  \param ref - reference points for phase shifting, list of floats in range from 0. to 1.
#  \param verb - verbose output, boolean
#  \param sve - save intermediate results, boolean
#  \retval phase - phase map, numpy float array
def get_phase_dev(dev, ref, verb = False, sve = False):

	start = time.time();

	if verb:
		print "get phase (device)"

	# get phase map
	p = dev.get_phase(ref, verb)

	time1 = time.time()

	if sve:
		if verb:
			print "saving results ..."
		dtmath.save_pgm('phase_dev.pgm', p, 1. / pi);

	if verb:
		print "acquisition time (ms):", int((time1 - start) * 1000)

	return p

## Unwrap phase map
#  
#  This function unwraps phase map and make normalization it lambda.
#
#  \param m - mask, numpy bool array
#  \param p - phase map, numpy float64 array
#  \param verb - verbose output, boolean
#  \retval surface (opd) - surface (opd), numpy float64 array
def unwrap_phase(m, p, verb = False):

	start = time.time();

	if verb:
		print "unwrap phase"
		
	s = dtmath.unwrap(uint8(m) * 255, float32(p))
	s = float64(s * .5 / pi)
	
	time1 = time.time()

	if verb:
		print "unwrapping time (ms):", int((time1 - start) * 1000)

	return s
	
# Display command line options
def usage():
	print """
phase [-h] [-d <device>] [-r <file>] [-o <file>] [-a <algorithm>] [-v] [-s]

    -h    print this message
    -d    device name
    -r    load normalized calibrated PZT voltages from file (use .npy file extension)
    -o    output results in file (use .npy file extension)
    -a    algorithm:
          0     capture frames from device and calculate phase map
          1     use device build in phase calulation algorithm
          2     compare result of both algorithms
    -v    verbose output
    -m    normalized mask radius
    -s    save intermediate results
 """

# Main
if __name__ == "__main__":
	#default parameters
	device = "dti"
	verb = False
	sve = False
#	ref = [ 0., 0.21214471, 0.37421732, 0.53773306, 0.70740915]
	ref = [ 0., 0.25, 0.5, 0.75, 1.0]
	alg = 2
	mask = 0.8
	fname = None

	# parse command line options
	try:
		opts, args = getopt.gnu_getopt(sys.argv[1:], "hd:r:o:vsa:r:")
	except getopt.error, msg:
		print msg
		print "for help use -h"
		sys.exit(2)

	# process options
	for o, a in opts:
		if o in ("-h"):
			usage()
			sys.exit(0)
		if o in ("-d"):
			device = a
		if o in ("-r"):
			# load reference in numpy format
			ref = load(a)
		if o in ("-o"):
			fname = a
		if o in ("-a"):
			alg = int(a)
			if alg < 0 or alg > 2:
				print "unsupported algorithm: " + str(alg)
				sys.exit(3)
		if o in ("-m"):
			mask = float(a)
			if mask < 0. or mask > 1.:
				print "invalid mask radius: " + str(mask)
				sys.exit(3)
		if o in ("-v"):
			verb = True
		if o in ("-s"):
			sve = True

	# imdev device
	dev = __import__(device + 'dev')
			
	# process get phase map
	if verb:
		print "parameters:", device + 'dev', verb, sve, ref

	if alg in (0, 2):
		# get phase map
		p1 = get_phase(dev, ref, verb, sve)
		out = p1
	if alg in (1, 2):
		# get phase map
		p2 = get_phase_dev(dev, ref, verb, sve)
		out = p2
	if alg == 2:
		# calculate error
		m = dtmath.roundmask(mask, p1.shape[1], p1.shape[0])
		opd1 = unwrap_phase(m, p1)
		opd2 = unwrap_phase(m, p2)
		e = opd1 - opd2
		out = e
		e = e * e;
		rms = pow(e.sum(), .5) / e.size
		print "RMS error:", rms
	
	# save results in numpy format
	if fname != None:
		print "saving results into " + fname + " ..."
		save(fname, out)

	sys.exit(0)

##
# @}

# EOF

