##############################################################################
## 
#  \file reports.py
#  \brief IAPACK reports.
#  \version 0.1
#  \author Dimitri Denk
#  \date 18.11.2011
#
#  This file provides various report creation functions
#
#  Copyright (c) 2010-2011
#  DENKTECH
#  www.denktech.de
#
#  Permission to use, copy, modify, distribute and sell this software
#  and its documentation for any purpose is hereby granted without fee,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear
#  in supporting documentation. DENKTECH makes no representations about 
#  the suitability of this software for any purpose. 
#  It is provided "as is" without express or implied warranty.
#
#############################################################################

##
# \addtogroup DTI_SCRIPTS scripts
# \ingroup DTI_TOOLS
#
# Python scripts for DTI interferometers
#
# @{

import sys
import time
import numpy as np
import math
from numpy import ma
import platform

from iapack import fast_zernike_coefficients, synthetic_surface, smooth

try:
	from reportlab.lib.enums import TA_JUSTIFY, TA_LEFT, TA_RIGHT, TA_CENTER
	from reportlab.lib.pagesizes import A4
	from reportlab.platypus import SimpleDocTemplate, Paragraph, Spacer, PageBreak, Table, TableStyle, Image
	from reportlab.lib.styles import getSampleStyleSheet, ParagraphStyle
	from reportlab.rl_config import defaultPageSize
	from reportlab.lib.units import inch, cm, mm
	from reportlab.lib import colors
	from reportlab.graphics.barcode.code128 import Code128
except ImportError as err:
	print err

try:
	import matplotlib.pyplot as plt
	import matplotlib.cm
	import matplotlib.mlab as mlab
	from matplotlib.figure import SubplotParams
except ImportError as err:
	print err

if sys.platform == "win32":
	import os
#	import Image
else:
	from StringIO import StringIO

Title = "Surface measurement report"
Wavelength = 532

class ImageList:
	def __init__(self, mem = True):
#		print '__init__'
		self.mem = mem
		self.idx = 0
	def __del__(self):
		if self.mem == False:
			for i in xrange(self.idx):
				fname = 'tmp' + str(i) + '.png'
				os.remove(fname)
	def make_image(self, fig):
		s = fig.get_size_inches()
		if self.mem:
			imgdata = StringIO()
			fig.savefig(imgdata, dpi=300, format='png', pad_inches=0., facecolor='r')
			imgdata.seek(0)
#			img = ImageReader(imgdata)
		else:
			imgdata = 'tmp' + str(self.idx) + '.png'
			fig.savefig(imgdata, dpi=300, format='png', pad_inches=0., facecolor='r')
			self.idx += 1
		img = Image(imgdata, s[0] * inch, s[1] * inch)

		return img
##
def zernikeTable(zk):
	colwidths = (2.5*cm, 2.*cm, 2.5*cm)
	rowheights = [.4*cm, .4*cm,]
	nzk = len(zk)
	data = [('Zernike coefficients',), ('coefficient', 'value', 'status',),]
	style = [
#		('GRID', (0,0), (-1,-1), 0.0, colors.black),
		('GRID', (0,0), (-1,-1), 0.0, colors.transparent),
		#('ALIGN', (0,0), (-1,-1), 'LEFT'),
		('ALIGN',(0,0),(-1,-1),'CENTER'),
		('VALIGN',(0,0),(-1,-1),'MIDDLE'),
		('HALIGN',(0,0),(-1,-1),'MIDDLE'),
		('LEFTPADDING',(0,0),(-1,-1), 0.),
		('RIGHTPADDING',(0,0),(-1,-1), 0.),
		('TOPPADDING',(0,0),(-1,-1), 0.),
		('BOTTOMPADDING',(0,0),(-1,-1), 0.),
		('FONT', (0,0), (-1,1), 'Helvetica-Bold', 10),
		('FONT', (0,2), (-1,-1), 'Helvetica', 10),
		('SPAN', (0, 0), (-1, 0)),
		('BACKGROUND', (0, 1), (-1, 1), colors.grey),
	]
	for i in xrange(nzk):
		rowheights.append(.4*cm)
		data.append((i, round(zk[i], 3), 'PASSED',))
		if i % 2:
			style.append(('BACKGROUND', (0, i + 2), (-1, i + 2), colors.lightgrey))
		if zk[i] > 0:
			style.append(('TEXTCOLOR',(1, i + 2),(1, i + 2), colors.darkred))
		elif zk[i] < 0:
			style.append(('TEXTCOLOR',(1, i + 2),(1, i + 2), colors.darkblue))
		else:
			style.append(('TEXTCOLOR',(1, i + 2),(1, i + 2), colors.black))

#		style.append(('TEXTCOLOR',(2, i + 2),(2, i + 2), colors.red))
		style.append(('TEXTCOLOR',(2, i + 2),(2, i + 2), colors.green))
	table = Table(data, colwidths, rowheights)
	table.setStyle(TableStyle(style))
	return table

def zernikeImage(il, zk):
	sp = SubplotParams(left=0.15, bottom=0.05, right=0.91, top=.9, wspace=0., hspace=0.)
	fig = plt.figure(figsize=(10.*cm/inch, 10.*cm/inch), frameon=False, subplotpars=sp)

	plt.bar(np.arange(len(zk))+.5, zk, align='center')
	plt.title('Zernike coefficients')
	plt.grid(True)
	pos = [-.5, 0.5, 5.5, 10.5, 15.5, 20.5, 25.5, 30.5, 35.5, 36.5]
	label = ['', '0', '5', '10', '15', '20', '25', '30', '35', '']
	plt.xticks(pos, label)
	
	return il.make_image(fig)

def dfactor(val, limit = 100.):
	v = abs(val)
	d = 1.
	if v < 1. and v != 0:
		d = int(math.floor(1. / v))
		if d > limit:
			d = limit
		if val > 0:
			return {'value':1. / d, 'label':r'$\mathbf{\lambda/%d}$' % d, 'label2':'\xce\xbb/%d' % d}
		else:
			return {'value':-1. / d, 'label':r'$\mathbf{-\lambda/%d}$' % d, 'label2':'-\xce\xbb/%d' % d}
	else:
		d = int(math.ceil(v))
		if val > 0:
			return {'value':d, 'label':r'$mathbf{%d\lambda}$' % d, 'label2':'%d\xce\xbb' % d}
		else:
			return {'value':-d, 'label':r'$mathbf{-%d\lambda}$' % -d, 'label2':'%d\xce\xbb' % d}

def mapImage(il, mask, data, title, limit):
	Zm = ma.masked_where(mask == False, data)

	sp = SubplotParams(left=0.02, bottom=0.02, right=0.98, top=.98, wspace=0., hspace=0.)
	fig = plt.figure(figsize=(8.5*cm/inch, 8.5*cm/inch), frameon=False, subplotpars=sp)

	plt.title(title)
	plt.axis('off')

	dfmax = dfactor(data.max(), limit)
	dfmin = dfactor(data.min(), limit)

	dmax = round(data.max(), 2)
	dmin = round(data.min(), 2)

	cmap = matplotlib.cm.jet
	CS = plt.imshow(Zm, cmap=cmap, interpolation='nearest', vmin=dfmin['value'], vmax=dfmax['value'])
	cbar = plt.colorbar(CS, shrink=0.77, ticks=[dfmin['value'], 0, dfmax['value'], dmin, dmax])
	cbar.ax.set_yticklabels([dfmin['label'], 0, dfmax['label'], '', ''])# vertically oriented colorbar

	return il.make_image(fig)

def statTable(v, name):
	colwidths = (2.5*cm, 2.1*cm, 2.1*cm, 2.5*cm)
	rowheights = [.4*cm, .4*cm,]
#	nzk = len(zk)
	data = [(name, ), ('parameter', '\xce\xbb', 'nm', 'status',),]
	style = [
#		('GRID', (0,0), (-1,-1), 0.0, colors.black),
		('GRID', (0,0), (-1,-1), 0.0, colors.transparent),
		('ALIGN',(0,0),(-1,-1),'CENTER'),
		('VALIGN',(0,0),(-1,-1),'MIDDLE'),
		('HALIGN',(0,0),(-1,-1),'MIDDLE'),
		('LEFTPADDING',(0,0),(-1,-1), 0.),
		('RIGHTPADDING',(0,0),(-1,-1), 0.),
		('TOPPADDING',(0,0),(-1,-1), 0.),
		('BOTTOMPADDING',(0,0),(-1,-1), 0.),
		('FONT', (0,0), (-1,1), 'Helvetica-Bold', 10),
		('FONT', (0,2), (-1,-1), 'Helvetica', 10),
		('SPAN', (0, 0), (-1, 0)),
		('BACKGROUND', (0, 1), (-1, 1), colors.grey),
	]

	values = [
		['min', .1, .023, 'PASSED'],
		['max', .1, .023, 'PASSED'],
		['pv', .1, .023, 'PASSED'],
		['rms', .1, .023, 'PASSED'],
	]
	values[0][1] = v.min()
	values[1][1] = v.max()
	values[2][1] = v.ptp()
#	values[3][1] = v.mean()
	values[3][1] = v.std()
	nv = len(values)
	for i in xrange(nv):
		values[i][2] = round(values[i][1] * Wavelength, 3)
		values[i][1] = round(values[i][1], 3)

	for i in xrange(nv):
		rowheights.append(.4*cm)
		data.append(values[i])
		if i % 2:
			style.append(('BACKGROUND', (0, i + 2), (-1, i + 2), colors.lightgrey))
		if values[i][1] > 0:
			style.append(('TEXTCOLOR',(1, i + 2),(2, i + 2), colors.darkred))
		elif values[i][1] < 0:
			style.append(('TEXTCOLOR',(1, i + 2),(2, i + 2), colors.darkblue))
		else:
			style.append(('TEXTCOLOR',(1, i + 2),(2, i + 2), colors.black))
#		style.append(('TEXTCOLOR',(2, i + 2),(2, i + 2), colors.red))
		style.append(('TEXTCOLOR',(3, i + 2),(3, i + 2), colors.green))

	table = Table(data, colwidths, rowheights)
	table.setStyle(TableStyle(style))
	return table


## Make report in PDF format
#
#  \param fname - file name, string
#  \param data - data to be saved, numpy array, list or tupple
def make_pdf_report(fname, data):
	print "making of PDF report..."

	noimg = False
	try:
		import _imaging
	except ImportError as err:
		print err
		print "WARNING: report donot support images. Use dti.py from command line to generate a report" 
		noimg = True
	
	print "calculating..."

	mask = data[0]
	opd = data[1]

	mopd = ma.masked_where(mask == False, opd)

	opd = mopd.__array__() * mask

	zk = fast_zernike_coefficients(mask, opd, 36)

	_zk = np.array(zk)
	tilt = synthetic_surface(zk[0:3], mask)

	error = opd - tilt

	error = smooth(mask, error, False)

	ferror = error

	_zk[0] = _zk[1] = _zk[2] = 0.
	fit = synthetic_surface(_zk, mask)

	fiterror = error - fit

	print "creating of PDF..."

	# image list
	il = ImageList(mem = (platform.system() == 'Linux'))

	idx = fname.rfind('.')
#	print "idx: " + str(idx)
	if idx < 0:
		fname += '.pdf'

	idx = fname.rfind('/')
#	print "idx: " + str(idx)
	if idx < 0:
		name = fname
	else:
		name = fname[idx + 1:]
	print "name: " + name
#	sn = name.upper()
	sn = name

	plt.rc('font', size = 10)
	plt.rcParams['mathtext.fontset'] = 'stixsans'

	doc = SimpleDocTemplate(fname, pagesize=A4,
		                    rightMargin=1.5*cm, leftMargin=2.5*cm,
		                    topMargin=1.*cm, bottomMargin=1.*cm)
	Story=[]

	barcode=Code128(sn, barWidth=0.5*mm, barHeight=20*mm)

	formatted_time = time.ctime()
	styles = getSampleStyleSheet()
	styleN = styles["Normal"]
	styleH = styles['Heading1']
	styles.add(ParagraphStyle(name='Center', alignment=TA_CENTER, fontSize = 12))

	label1 = dfactor(error.max() - error.min())['label2']
	label2 = dfactor(fit.max() - fit.min())['label2']

	stat1 = statTable(error, "Error")
	stat2 = statTable(fit, "Fit")

	table = zernikeTable(zk)

	if noimg:
		img = Paragraph('No image', styles["Center"])
	else:
		img = zernikeImage(il, _zk)

	h = .4 * (len(zk) + 1) * cm
	colwidths = (10.*cm, 7.*cm)
	rowheights = (3. * cm, 6 * .4 * cm, 7 * .4 * cm, 10.*cm)
	data = (
		((
			Paragraph(Title, styleH),
			Paragraph('Device: DTI2', styles["Normal"]),
			Paragraph('Wavelength: ' + str(Wavelength) + 'nm', styles["Normal"]),
			Paragraph('Surface quality: ' + label1 + ' (' + label2 + ')', styles["Normal"]),
			Paragraph('Time: ' + formatted_time, styles["Normal"]),
			),
			(barcode, Paragraph('ID: ' + sn, styles["Center"]), ),
		),
		(stat1, table,),	
		(stat2, '2',),	
		(img, '3',),
	)
	table = Table(data, colwidths, rowheights)
	style = TableStyle([
#	('GRID', (0,0), (-1,-1), 0.0, colors.black),
	('GRID', (0,0), (-1,-1), 0.0, colors.transparent),
	#('ALIGN', (0,0), (-1,-1), 'LEFT'),
	('ALIGN',(0,0),(-1,-1),'CENTER'),
	('VALIGN',(0,0),(-1,-1),'MIDDLE'),
	('LEFTPADDING',(0,0),(-1,-1), 0.),
	('RIGHTPADDING',(0,0),(-1,-1), 0.),
	('TOPPADDING',(0,0),(-1,-1), 0.),
	('BOTTOMPADDING',(0,0),(-1,-1), 0.),
	('SPAN', (1,1), (1,3)),
	('ALIGN', (0,0), (0,2), 'LEFT'),
	('VALIGN',(0,1),(0,2),'BOTTOM'),
	('VALIGN',(0,0),(-1,0),'TOP'),
	])
	table.setStyle(style)
	Story.append(table)

	if noimg:
		img1 =  Paragraph('No image', styles["Center"])
		img2 =  Paragraph('No image', styles["Center"])
	else:
		img1 = mapImage(il, mask, error, "Surface error", 20.)
		img2 = mapImage(il, mask, fit, "Surface fit", 100.)
	
	colwidths = (8.5*cm, 8.5*cm)
	rowheights = (8.5*cm)
	data = (
		(img1, img2,),	
#		(None, None,),	
	)
	style = TableStyle([
#	('GRID', (0,0), (-1,-1), 0.0, colors.black),
	('GRID', (0,0), (-1,-1), 0.0, colors.transparent),
	('ALIGN',(0,0),(-1,-1),'CENTER'),
	('VALIGN',(0,0),(-1,-1),'MIDDLE'),
	('LEFTPADDING',(0,0),(-1,-1), 0.),
	('RIGHTPADDING',(0,0),(-1,-1), 0.),
	('TOPPADDING',(0,0),(-1,-1), 0.),
	('BOTTOMPADDING',(0,0),(-1,-1), 0.),
	])
	table = Table(data, colwidths, rowheights)
	table.setStyle(style)
	Story.append(table)

	Story.append(Paragraph('<font size=8>&copy; DENKTECH. All rights reserved. IAPACK report generator v0.1</font>', styles["Normal"]))

	doc.build(Story)

	print "report was saved as: \"" + fname + "\"."

# @}

# EOF

