##############################################################################
## 
#  \file simdev.py
#  \brief DTI device simulator.
#  \version 0.2
#  \author Dimitri Denk
#  \date 12.08.2011
#
#  This file provides DTI device simulator communication functions.
#
#  Copyright (c) 2010-2011
#  DENKTECH
#  www.denktech.de
#
#  Permission to use, copy, modify, distribute and sell this software
#  and its documentation for any purpose is hereby granted without fee,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear
#  in supdeving documentation. DENKTECH makes no representations about 
#  the suitability of this software for any purpose. 
#  It is provided "as is" without express or implied warranty.
#
##############################################################################

##
# \addtogroup DTI_SIM_DEVICE DTI device simulator
# \ingroup DTI_TOOLS
#
# Python scripts for DTI interferometers
#
# @{

# 

from numpy import *
import dtmath
from math import ceil, pi as PI
import time

last_shift = 0.
last_time = time.time();

## Get frame with phase shifting
#  
#  This function captures one frame with desired phase shifting
#
#  \param shift - normalized PZT voltage, float in range from 0. to 1. or None.
#  If shift is None, function will capture frame without phase shifting.
#  \param verb - verbose output, boolean
#  \retval frame - image, numpy uint16 array
def get_frame(shift, verb = False):

	start = time.time();

	if shift == None:
		shift = 0

	global last_shift, last_time

	_shift = shift - (shift - last_shift) * .01 / (start - last_time) + random.random() * .01

#	z = array([.1, -1.5, 1.0, -0.1, .0, .0, -.05,])
#	z = array([.1, -1.5])
	z = array([.1, 1.5, 0.0, -0.1, .0, .0, .05,])
	m = dtmath.roundmask(.99, 512, 512)
	s = dtmath.zernike_surface(z, m)
	f = dtmath.interferogram(2. * PI, 2. * PI * (_shift + .2) * 1.41, 32768, m, s)
#	f += 16386
	f += 11500
	noise = random.random(f.shape)
	noise *= .25 * 32768
	f = f + uint16(noise)
#	time.sleep(.02)

	time1 = time.time()

	if verb:
		print "get_frame with shift " + str(shift) + ", time " + str(int((time1 - start) * 1000 + .5)) + "ms"

	last_shift = shift
	last_time = start

	return f

## Get phase map from device
#  
#  This function gets phase map using device build-in phase calculation
#  algorithm
#
#  \param ref - reference points for phase shifting, list of floats in range
#  from 0. to 1.
#  \param verb - verbose output, boolean
#  \retval phase - phase map, numpy float64 array
def get_phase(ref, verb = False):

	start = time.time();

	i = [];
	for r in ref:
		i.append(get_frame(r));
	p = dtmath.phase5(i)
	p = float64(p * PI / 32768)
	time1 = time.time()
	if verb:
		print "get_phase time " + str(int((time1 - start) * 1000 + .5)) + "ms"

	return p

##
# @}

# EOF

