##############################################################################
## 
#  \file zernike.py
#  \brief Zernike polynomial coefficients.
#  \version 0.2
#  \author Dimitri Denk
#  \date 12.04.2011
#
#  This file provides Zernike polynomial coefficients calculation functions
#
#  Copyright (c) 2010-2011
#  DENKTECH
#  www.denktech.de
#
#  Permission to use, copy, modify, distribute and sell this software
#  and its documentation for any purpose is hereby granted without fee,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear
#  in supporting documentation. DENKTECH makes no representations about 
#  the suitability of this software for any purpose. 
#  It is provided "as is" without express or implied warranty.
#
#############################################################################

##
# \addtogroup DTI_SCRIPTS scripts
# \ingroup DTI_TOOLS
#
# Python scripts for DTI interferometers
#
# @{

from numpy import *
import time
# import DENKTECH libraries
import dtmath

## Calculate Zernike coefficients
#
#  This function calculates Zernike coefficients for a given surface
#
#  \param m - mask, numpy bool array
#  \param s - surface (opd), numpy float array
#  \param n - number of polynomials, positive integer
#  \retval z - Zernike coefficients, numpy bool array
def zernike_coefficients(m, s, n):
	for i in xrange(n):
		K = zeros(n)
		K[i] = 1.0
		S = dtmath.zernike_surface(K, m)
		S = S.reshape(S.size)
		if i == 0:
			a = S
		else:
			a = append(a, S)
	a = a.reshape(n, S.size)
	a = a.transpose()
	A = linalg.pinv(a)
	S = s.reshape(s.size)
	K = dot(A, S)
	return K

## Calculate Zernike coefficients 
#
#  This function tries first to reduce data size and then it calculates
#  Zernike coefficients.
#
#  \param m - mask, numpy bool array
#  \param s - surface (opd), numpy float array
#  \param n - number of polynomials, positive integer
#  \param verb - verbose output, boolean
#  \retval z - Zernike coefficients, numpy bool array
def fast_zernike_coefficients(m, s, n, verb = False):
	start = time.time();
	w = m.shape[1]
	if(w < 64):
		sk = 1
	else:
		for sk in xrange(w / 64 + 1, 0, -1):
			if(w % sk == 0):
				break
	if verb:
		print "scale factor: " + str(sk)
	sopd = dtmath.binning(s, sk) / (sk * sk)
	smask = dtmath.binning(float64(m), sk)
	smask = dtmath.make_mask(smask, sk * sk)
	k = zernike_coefficients(smask, sopd, n)
	time1 = time.time()
	if verb:
		print "zernike coefficients calculation time " + str(int((time1 - start) * 1000 + .5)) + "ms"
	return k

## Generate synthetic surface 
#
#  This function generates a synthetic surface using Zernike coefficients
#  and mask
#
#  \param z - Zernike coefficients, numpy float64
#  \param m - mask, numpy bool array
#  \param verb - verbose output, boolean
#  \retval s - surface (opd), numpy float64 array
def synthetic_surface(z, m, verb = False):
	start = time.time();
	s = dtmath.zernike_surface(z, m)
	time1 = time.time()
	if verb:
		print "synthetic surface calculation time " + str(int((time1 - start) * 1000 + .5)) + "ms"
	return s
	
if __name__ == "__main__":
	print "Zernike coeffizients calculation example"
	Z = array([.1, -1.5, 1.0, -0.1, .0, .0, -.05,])
	m = dtmath.roundmask(.99, 512, 512)
	s = dtmath.zernike_surface(Z, m)
	start = time.time();
	z = zernike_coefficients(m, s, 16)
	time1 = time.time();
	fz = fast_zernike_coefficients(m, s, 16)
	time2 = time.time();
	print "original Zernike = " + str(Z)
	print "standard Zernike = " + str(z)
	print "time: " + str()
	print "fast Zernike = " + str(z)

##
# @}

# EOF
