# !/usr/bin/python3
##############################################################################
##
#  \file dti.py
#  \brief DTI main script and interface functions.
#  \version 0.5
#  \author Dimitri Denk
#  \date 12.08.2011
#
#  This file provides opd calculation functions
#
#  Copyright (c) 2010-2021
#  DENKTECH
#  http://www.denktech.de
#
#  Permission to use, copy, modify, distribute and sell this software
#  and its documentation for any purpose is hereby granted without fee,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear
#  in supporting documentation. DENKTECH makes no representations about
#  the suitability of this software for any purpose.
#  It is provided "as is" without express or implied warranty.
#
#  History
#  12.08.2011 DD first release
#  04.08.2021 DD revorking of code for Python3
#
#############################################################################

##
# \addtogroup DTI_SCRIPTS scripts
# \ingroup DTI_TOOLS
#
# Python scripts for DTI interferometers
#
# @{

import os
import sys
import numpy as np
import getopt

# import DENKTECH libraries
import dtmath

# import DENKTECH scripts
import iapack
import iapack.reports as reports

# Remove '' and current working directory from the first entry
# of sys.path, if present to avoid using current directory
# in pip commands check, freeze, install, list and show,
# when invoked as python -m pip <command>
if sys.path[0] in ('', os.getcwd()):
    sys.path.pop(0)

# If we are running from a wheel, add the wheel to sys.path
# This allows the usage python pip-*.whl/pip install pip-*.whl
if __package__ == '':
    # __file__ is pip-*.whl/pip/__main__.py
    # first dirname call strips of '/__main__.py', second strips off '/pip'
    # Resulting path is the name of the wheel itself
    # Add that to sys.path so we can import pip
    path = os.path.dirname(os.path.dirname(__file__))
    sys.path.insert(0, path)


# Display command line options
def _usage_():
    print("""
-m iapack <command> [options...]

  command          frame|calibrate|phase|phasedev|opd

  common options
    -h, --help     print(this message
    --device       device name
    --save-prefix  save file prefix
    --save-data    save data (npy and pgm file formats)
    --save-error   save error (text file format)
    --save-all     save all intermediate results
    -n             repeat command
    -v             verbose output

  commsnd-dependent options
    -a             make data averaging
    --pzt-max      normalized maximal PZT voltage in range from 0. to 1.
    --pzt-ref      load normalized calibrated PZT voltages from file
                   (use .npy file extension)
    --mask         load mask from file (use .npy file extension)
    --zernike      calculate Zernike polynomial coefficients

  capture frame
    command        frame
    options        -a, <common options>

  calibrate PZT
    command        calibrate
    options        --pzt-max, <common options>

  capture frames and calculate phase map
    command        phase
    options        --pzt-ref, <common options>

  get phase map from device (DTI build-in algorithm)
    command        phasedev
    options        --pzt-ref, <common options>

  calculate opd (surface reconstruction)
    command        opd
    options        -a, --pzt-ref, --mask, --zernike, <common options>

  generate report from saved opd
    command        report
    options        none
 """)
    pargs = (str(sys.executable), ) * 6
    print("""  examples
    %s -m iapack frame --save-data
    %s -m iapack calibrate --pzt-max=0.85 -a5 --save-data
    %s -m iapack phase --pzt-ref=test_calibrate_0.npy --save-data
    %s -m iapack phasedev --pzt-ref=test_calibrate_0.npy --save-data
    %s -m iapack opd --pzt-ref=test_calibrate_0.npy -n4 --save-data --save-error
    %s -m iapack report test_opd_0.npy test_opd_0.pdf
""" % pargs)


# Main
if __name__ == "__main__":

    print("iapack")

    # default parameters
    verb = False
    sve = False
    ref = [0., 0.21214471, 0.37421732, 0.53773306, 0.70740915]
    cmd = -1
    mask = None
    ref_fname = None
    mask_fname = None
    save_prefix = "test"
    save_data = False
    save_error = False
    zn = 0
    avr = 1
    rep = 1
    pztmax = 0.85
    points = 10
    shift = None
    errmon = True
    # device = 'DTISIM'
    device = 'DTI3'

    # parse command line options
    try:
        opts, args = getopt.gnu_getopt(sys.argv[1:], "hva:n:",
                                       ["help", "device=", "pzt-ref=", "mask=", "zernike=", "save-data", "save-all",
                                        "save-prefix=", "save-error", "pzt-max="])
    except getopt.error as msg:
        print(msg)
        print("for help use -h")
        sys.exit(1)

    # process options
    for o, a in opts:
        if o in ("-h", "--help"):
            _usage_()
            sys.exit(0)
        if o in ("--device",):
            device = a
        if o in ("-n",):
            rep = int(a)
        if o in ("--pzt-ref",):
            ref_fname = a
        if o in ("--mask",):
            mask_fname = a
        if o in ("--save-prefix",):
            save_prefix = a
        if o in ("--save-data",):
            save_data = True
        if o in ("--save-error",):
            save_error = True
        if o in ("-a",):
            avr = int(a)
        if o in ("--zernike",):
            zn = int(a)
        if o in ("--pzt-max",):
            pztmax = float(a)
        if o in ("-v",):
            verb = True
        if o in ("--save-all",):
            sve = True
            save_data = True
            save_error = True

    # process arguments
    argc = len(args)
    if argc > 0:
        cmd = args[0]
        if cmd == "frame":
            ref_fname = None
            zn = 0
        # calibrate
        elif cmd == "calibrate":
            ref_fname = None
            zn = 0
        # get phase
        elif cmd == "phase":
            avr = 1
            errmon = False
            zn = 0
        # get phase (device build-in algorithm)
        elif cmd == "phasedev":
            avr = 1
            errmon = False
            zn = 0
        # compare two phase algorithms
        elif cmd == "phasecmp":
            avr = 1
            errmon = False
            zn = 0
        # calculate opd
        elif cmd == "opd":
            pass
        # make report
        elif cmd == "report":
            # process arguments
            if argc > 2:
                fname = args[1]
                opd = iapack.load(fname)
                mask = dtmath.make_mask(opd[0], .5)
                opd = opd[1]
                reports.make_pdf_report(args[2], [mask, opd])
            sys.exit(0)
        # make report2
        elif cmd == "report2":
            # process arguments
            if argc > 2:
                fname = args[1]
                reports.make_pdf_report2(args[2], fname)
                sys.exit(0)
            else:
                print("too few arguments")
                sys.exit(4)
        else:
            print("unsupported command \"" + args[0] + "\"")
            sys.exit(3)
    else:
        print("command is not specified")
        print("for help use -h")
        sys.exit(2)

    # load referense in numpy format
    if ref_fname != None:
        ref = iapack.load(ref_fname)

    # load mask in numpy format
    if mask_fname != None:
        mask = iapack.load(mask_fname)

    # create device
    devclass = iapack.__dict__[device]
    dev = devclass()

    # make pass
    datalist = []
    for t in range(rep):
        print("pass " + str(t + 1) + " of " + str(rep))

        avrdatalist = []
        for i in range(avr):
            if verb is False:
                sys.stdout.write(".")
                sys.stdout.flush()
            else:
                print("iteration " + str(i + 1) + " of " + str(avr))

            if cmd == "frame":
                data = iapack.phase.get_frame(dev, shift=shift, verb=verb)
            elif cmd == "phase":
                data = iapack.phase.get_phase(dev, ref, verb, sve)
            elif cmd == "phasedev":
                data = iapack.phase.get_phase_dev(dev, ref, verb, sve)
            elif cmd == "phasecmp":
                if verb:
                    print("compare phase calculation algorithms")
                data = np.zeros(1)
            # calculate opd
            elif cmd == "opd":
                phasemap = iapack.phase.get_phase_dev(dev, ref, verb, sve)
                # phasemap = phase.get_phase(dev, ref, verb, sve)
                # check mask
                if mask is None:
                    mask = dtmath.roundmask(.8, phasemap.shape[1], phasemap.shape[0])
                # calculate opd
                opd = iapack.phase.unwrap_phase(mask, phasemap)
                data = opd
            elif cmd == "calibrate":
                if verb:
                    print(pztmax, points)
                data = iapack.calibrate(dev, umax=pztmax, points=points, samples=2, verb=verb, sve=sve)
            else:
                sys.stderr.write("unkmown command")
                sys.exit(3)

            # store result
            avrdatalist.append(data.astype(np.double))

            # averaging
            if i == 0:
                avrdata = data.astype(np.double)
            else:
                avrdata += data.astype(np.double)

        if verb is False:
            print("")

        # calculate results
        avrdata /= avr

        # calculate deviation
        if avr > 1:
            for data in avrdatalist:
                e = data - avrdata
                e = e * e;
                rmsval = pow(e.sum(), .5) / e.size
                print("rms deviation: " + str(rmsval))
        #		if rms > 1e-3:
        #			print("OPD error too big: " + str(rms)

        # print(result
        if cmd == "calibrate":
            print("calibrated values: ", avrdata)

        # append result
        datalist.append(avrdata)

        # calculate zernike coefficients
        if zn > 0:
            zk = iapack.zernike.zernike_coefficients(mask, avrdata, zn)
            # output results
            print("zernike coefficients: " + str(zk))

    # calculate deviation
    if errmon:
        errlist = []
        for data in datalist:
            e = data - datalist[0]
            e = e * e;
            rmsval = pow(e.sum(), .5) / e.size
            print("rms error: " + str(rmsval))
            errlist.append(rmsval)

    # save results
    if save_data:
        for i in range(rep):
            if cmd == "calibrate":
                dat_name = save_prefix + "_" + cmd + "_" + str(i) + ".dat"
                print("saving results into " + dat_name)
                iapack.utils.save_txt(dat_name, datalist[i])
                npy_name = save_prefix + "_" + cmd + "_" + str(i) + ".npy"
                print("saving results into " + npy_name)
                iapack.save(npy_name, datalist[i])
            else:
                pgm_name = save_prefix + "_" + cmd + "_" + str(i) + ".pgm"
                npy_name = save_prefix + "_" + cmd + "_" + str(i) + ".npy"
                print("saving results into " + pgm_name)
                print("saving results into " + npy_name)
                if cmd == "frame":
                    dtmath.save_pgm(pgm_name, datalist[i].astype(np.uint16));
                    iapack.save(npy_name, datalist[i].astype(np.uint16))
                elif cmd in ("phase", "phasedev"):
                    k = 2 * np.pi
                    dtmath.save_pgm(pgm_name, datalist[i], 1. / np.pi);
                    iapack.save(npy_name, datalist[i].astype(np.uint16))
                elif cmd == "opd":
                    dtmath.save_pgm(pgm_name, datalist[i], .1);
                    iapack.save(npy_name, [mask, datalist[i]])
                else:
                    dtmath.save_pgm(pgm_name, datalist[i], .1);
                    iapack.save(npy_name, datalist[i])

    # save error
    if save_error:
        if errmon:
            dat_name = save_prefix + "_" + cmd + "_err.npy"
            print("saving results into " + dat_name)
            iapack.save(dat_name, errlist)
            dat_name = save_prefix + "_" + cmd + "_err.dat"
            print("saving results into " + dat_name)
            iapack.utils.save_txt(dat_name, errlist)

    sys.exit(0)

##
# @}

# EOF

