##############################################################################
## 
#  \file dti3.py
#  \brief DTI device.
#  \version 0.4
#  \author Dimitri Denk
#  \date 04.02.2012
#
#  This file provides communication interface for DTI device.
#
#  Copyright (c) 2010-2021
#  DENKTECH
#  http://www.denktech.de
#
#  Permission to use, copy, modify, distribute and sell this software
#  and its documentation for any purpose is hereby granted without fee,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear
#  in supporting documentation. DENKTECH makes no representations about
#  the suitability of this software for any purpose. 
#  It is provided "as is" without express or implied warranty.
#
#  History
#  04.02.2012 DD interface optimization
#  24.07.2021 DD reworking of code for Python3
#
##############################################################################

##
# \addtogroup DTI_DEVICE DTI3 device
# \ingroup DTI_TOOLS
#
# Python scripts for DTI interferometers
#
# @{

import time
import requests
import numpy as np
from typing import Optional, List


class DTI3:
	def __init__(self):
		self.last_shift = 0.
		self.last_time = time.time()
		# self.ip = "192.168.3.102"
		self.ip = "192.168.3.33"
		self.port = 80

	def _get_pgm(self, uri: str, verb: bool = False, timeout: float = 1.0) -> np.ndarray:
		_uri_ = "http://%s:%d/%s" % (self.ip, self.port, uri, )
		if verb:
			print("uri:", _uri_)
		r = requests.get(_uri_, timeout=timeout)
		if verb:
			print("request result", r)
		if r.status_code != 200:
			raise IOError("status: %d" % r.status_code)
		try:
			if len(r.content) < 15:
				raise ValueError("wrong header size")
			elif r.content[0:2] != b'P5\n':
				# print(r.content[0:22])
				idx1 = r.content.find(b' ', 3)
				idx2 = r.content.find(b'\n', idx1 + 1)
				idx3 = r.content.find(b'\n', idx2 + 1)
				# print(idx1, idx2, idx3)
				width = int(r.content[3:idx1])
				height = int(r.content[idx1 + 1:idx2])
				maxval = int(r.content[idx2 + 1:idx3])
				if verb:
					print("P5", width, height, maxval)
		except AttributeError:
			raise ValueError("content is not PGM file")
		# make numpy array
		return np.frombuffer(
			r.content,
			dtype='u1' if int(maxval) < 256 else '>u2',
			count=int(width) * int(height),
			offset=idx3 + 1
			).reshape((int(height), int(width)))

	## Get frame with phase shifting
	#  
	#  This function captures one frame with desired phase shifting
	#
	#  \param shift - normalized PZT voltage, float in range from 0. to 1. or None.
	#  If shift is None, function will capture frame without phase shifting.
	#  \param verb - verbose output, boolean
	#  \retval frame - image, numpy uint16 array
	def get_frame(self, shift: Optional[float], verb: bool = False) -> np.ndarray:
		if verb:
			print("enter get_frame")
		if shift != None:
			uri = "img.pgm?shift=%d" % int(shift * 65535)
		else:
			uri = "img.pgm"
		return self._get_pgm(uri, verb, timeout=0.5)

	## Get phase map from device
	#  
	#  This function gets phase map using device build-in phase calculation
	#  algorithm
	#
	#  \param ref - reference points for phase shifting, list of floats in range
	#  from 0. to 1.
	#  \param verb - verbose output, boolean
	#  \retval phase - phase map, numpy float64 array
	def get_phase(self, ref: List[float], verb: bool = False) -> np.ndarray:
		# prepare uri
		uri = "phase.pgm?"
		for i in range(len(ref)):
			if i > 0:
				uri = uri + "&"
			uri = uri + "shift=" + str(int(ref[i] * 65535))
		p = self._get_pgm(uri, verb, timeout=3.)
		p = np.float64(np.pi - (p * np.pi / 32768))

		return p

	## Set device parameter
	#
	#  This function sets device parameter
	#
	#  \param name - parameter name, string.
	#  \param val - parameter value, depends on parameter type
	#  \retval None or exception
	def set_param(self, name: str, value: str) -> None:
		print("parameter '" + name + "', value '" + value + "'")
		if name == "ip":
			self.ip = value
		elif name == "port":
			self.port = int(value)
		else:
			print("unknown parameter '" + name + "'")

##
# @}


if __name__ == "__main__":
	dev = DTI3()
	frame = dev.get_frame(shift=0.1, verb=True)
	print(frame)

# EOF

