##############################################################################
## 
#  \file simdev.py
#  \brief DTI device simulator.
#  \version 0.4
#  \author Dimitri Denk
#  \date 12.08.2011
#
#  This file provides DTI device simulator communication functions.
#
#  Copyright (c) 2010-2021
#  DENKTECH
#  http://www.denktech.de
#
#  Permission to use, copy, modify, distribute and sell this software
#  and its documentation for any purpose is hereby granted without fee,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear
#  in supdeving documentation. DENKTECH makes no representations about 
#  the suitability of this software for any purpose. 
#  It is provided "as is" without express or implied warranty.
#
#  History
#  12.08.2011 DD first release
#  24.07.2021 DD reworking of code for Python3
#
##############################################################################

##
# \addtogroup DTI_SIM_DEVICE DTI device simulator
# \ingroup DTI_TOOLS
#
# Python scripts for DTI interferometers
#
# @{

# 

import numpy as np
import dtmath
import time
from typing import Optional, List


class DTISIM:
	def __init__(self):
		self.last_shift = 0.
		self.last_time = time.time()

	## Get frame with phase shifting
	#
	#  This function captures one frame with desired phase shifting
	#
	#  \param shift - normalized PZT voltage, float in range from 0. to 1. or None.
	#  If shift is None, function will capture frame without phase shifting.
	#  \param verb - verbose output, boolean
	#  \retval frame - image, numpy uint16 array
	def get_frame(self, shift: Optional[float], verb: bool = False) -> np.ndarray:

		start = time.time()
		# if (start - self.last_time) < 0.08:
		# print("sleep...", 0.08 - (start - self.last_time))
		# time.sleep(0.08 - (start - self.last_time))
		# start = time.time();

		if shift is None:
			shift = 0

		# _shift = shift - (shift - self.last_shift) * .001 / (start - self.last_time) + random.random() * .001
		_shift = shift - (shift - self.last_shift) * .01 + np.random.random() * .001

		# initial surface
		z = np.array([.1, 1.5, 0.0, -0.1, .0, .0, .05, ])
		m = dtmath.roundmask(.95, 512, 512)
		s = dtmath.zernike_surface(z, m)
		f = dtmath.interferogram(2. * np.pi, 2. * np.pi * (_shift + .2) * 1.33333, 32768, m, s)
		# intencity offset
		f += 11500
		# noise
		noise = np.random.random(f.shape)
		noise *= .25 * 32768
		# appy mask
		f = (f + np.uint16(noise)) * m

		time1 = time.time()

		if verb:
			print("get_frame with shift %.3f, time %.1fms" % (shift, int((time1 - start) * 1000 + .5)))

		self.last_shift = shift
		self.last_time = start

		return f

	## Get phase map from device
	#
	#  This function gets phase map using device build-in phase calculation
	#  algorithm
	#
	#  \param ref - reference points for phase shifting, list of floats in range
	#  from 0. to 1.
	#  \param verb - verbose output, boolean
	#  \retval phase - phase map, numpy float64 array
	def get_phase(self, ref: List[float], verb: bool = False) -> np.ndarray:

		start = time.time()

		i = []
		for r in ref:
			i.append(self.get_frame(r))
		p = dtmath.phase5(i)
		p = np.float64(p * np.pi / 32768)
		time1 = time.time()
		if verb:
			print("get_phase time " + str(int((time1 - start) * 1000 + .5)) + "ms")

		return p

	## Set device parameter
	#
	#  This function sets device parameter
	#
	#  \param name - parameter name, string.
	#  \param val - parameter value, depends on parameter type
	#  \retval None or exception
	def set_param(self, name: str, value: str) -> None:
		self.last_time = time.time()
		print("set_param(%s, %s)" % (name, value, ))
		pass

##
# @}

# EOF
