/****************************************************************************/
/**
 * \file dtcam.c
 * \brief Python image capture package for DTI interferometers
 * \version 0.3
 * \author Dimitri Denk
 * \date 02.02.2010
 *
 * This file implements DTI communication class.
 *
 * Copyright (c) 2010 
 * DENKTECH
 * www.denktech.de
 *
 * Permission to use, copy, modify, distribute and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation. DENKTECH makes no representations about 
 * the suitability of this software for any purpose. 
 * It is provided "as is" without express or implied warranty.
 */
/****************************************************************************/

#include "Python.h"
#include "numpy/arrayobject.h"

#ifdef _WIN32
#include <winsock.h>
#else
#include <arpa/inet.h>
#endif

#include "dtcam.h"

#define MODULE_VERSION "MAJOR_VERSION.MINOR_VERSION"

/* PORT */
typedef struct {
	PyObject_HEAD
	unsigned long ip;
	unsigned short port;
} PyDtcam;

static int _dtcamport_init(PyDtcam *self, PyObject *args, PyObject *kwds)
{
	return 0;
}

static PyObject* _dtcamport_new(PyTypeObject *type, PyObject *args, PyObject *kwds)
{
	PyDtcam *self;

	const char *ip = "127.0.0.1";
	unsigned short port = 5557;
	static char *keywords[] = {"ip", "port", NULL};
	if (!PyArg_ParseTupleAndKeywords(args, kwds, "|sH", keywords, &ip, &port))
		return NULL;
	self = (PyDtcam*)type->tp_alloc(type, 0);
	if (self != NULL)
	{
		self->ip = inet_addr(ip);
		self->port = port;
	}
	return (PyObject*)self;
}

static void _dtcamport_dealloc(PyDtcam* self)
{
	self->ob_type->tp_free(self);
}

static PyObject* _dtcamport_get_frame(PyDtcam* self, PyObject *args)
{
	int result;
	unsigned char *buf;
	size_t size = 1280 * 1024 * 2 + 256;
	unsigned short width, height;
	unsigned char bpp;
	const char* uri = "/img.pgm";
	size_t err_size = 1024;
	char *err_buf;
	unsigned char type;
	PyObject *retval = NULL;

	if (!PyArg_ParseTuple(args, "|s:get_frame", &uri)) {
		return NULL;
	}

//	fprintf(stdout, "_dtcamport_get_frame: %s\n", uri);

	buf = (unsigned char*)malloc(size);
	err_buf = (char*)malloc(err_size);
	if(buf && err_buf) {
		result = perform_http_request(self->ip, self->port, uri, buf, size, err_buf, err_size);
		if(result > 0) {
			result = process_http_response(buf, result, &type, err_buf, err_size);
			if(result >= 0) {
				if(type == 3) {
					result = extract_pgm(buf, result, &width, &height, &bpp, err_buf, err_size);
					if(result >= 0) {
						PyArrayObject *frame;
						int dimensions[2];
						dimensions[1] = width;
						dimensions[0] = height;
						frame = (PyArrayObject*)PyArray_SimpleNew(2, dimensions, NPY_USHORT);
						if(frame) {
							unsigned long i;
							npy_ushort *dest;
							size = width * height;
							dest = (npy_ushort*)frame->data;
							if(bpp > 8) {
								const unsigned short *src = (const unsigned short*)buf;
								for(i = 0; i < size; i++) {
									*dest = (*src) << (16 - bpp);
									src++;
									dest++;
								}
								result = 0;
							}
							else {
								const unsigned char *src = (const unsigned char*)buf;
								for(i = 0; i < size; i++) {
									*dest = ((unsigned short)(*src)) << (16 - bpp);
									src++;
									dest++;
								}
							}
							retval = PyArray_Return(frame);
						}
						else {
							PyErr_SetString(PyExc_MemoryError, "cannot allocate array");
						}
					}
					else {
						PyErr_SetString(PyExc_RuntimeError, err_buf);
					}
				}
				else {
					PyErr_SetString(PyExc_RuntimeError, "wrong content type");
				}
			}
			else {
				PyErr_SetString(PyExc_RuntimeError, err_buf);
			}
		}
		else {
			PyErr_SetString(PyExc_RuntimeError, err_buf);
		}
	}
	else {
		retval = PyErr_Format(PyExc_RuntimeError, "cannot allocate memory");
	}
	if(buf) {
		free(buf);
	}
	if(err_buf) {
		free(err_buf);
	}

	return retval;
}

static PyMethodDef dtcamport_methods[] = {
	{"get_frame", (PyCFunction)_dtcamport_get_frame, METH_VARARGS, "get frame."},
    {NULL}  /* Sentinel */
};

static PyTypeObject PyDtcamType = {
    PyObject_HEAD_INIT(NULL)
    0,                         /*ob_size*/
    "dtcamport",               /*tp_name*/
    sizeof(PyDtcam),       /*tp_basicsize*/
    0,                         /*tp_itemsize*/
    (destructor)_dtcamport_dealloc,	/*tp_dealloc*/
    0,                         /*tp_print*/
    0,                         /*tp_getattr*/
    0,                         /*tp_setattr*/
    0,                         /*tp_compare*/
    0,                         /*tp_repr*/
    0,                         /*tp_as_number*/
    0,                         /*tp_as_sequence*/
    0,                         /*tp_as_mapping*/
    0,                         /*tp_hash */
    0,                         /*tp_call*/
    0,                         /*tp_str*/
    0,                         /*tp_getattro*/
    0,                         /*tp_setattro*/
    0,                         /*tp_as_buffer*/
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE, /*tp_flags*/
    "DTCAMPORT object",           /* tp_doc */
    0,		               /* tp_traverse */
    0,		               /* tp_clear */
    0,		               /* tp_richcompare */
    0,		               /* tp_weaklistoffset */
    0,		               /* tp_iter */
    0,		               /* tp_iternext */
    dtcamport_methods,             /* tp_methods */
    0,//Noddy_members,             /* tp_members */
    0,//Noddy_getseters,           /* tp_getset */
    0,                         /* tp_base */
    0,                         /* tp_dict */
    0,                         /* tp_descr_get */
    0,                         /* tp_descr_set */
    0,                         /* tp_dictoffset */
    (initproc)_dtcamport_init,         /* tp_init */
    0,                         /* tp_alloc */
    _dtcamport_new,                   /* tp_new */
};

/* MODULE */

static PyMethodDef py_methods[] = {
	{NULL, NULL, 0, NULL}        /* Sentinel */
};

PyMODINIT_FUNC initdtcam(void)
{
	PyObject *module;

#ifdef _WIN32
	/* initialize Winsock library */
	WSADATA wsaData;
	WORD wVersionRequested = MAKEWORD(1, 1);
	int result = WSAStartup(wVersionRequested, &wsaData);
	if(result != 0)
	{
		fprintf(stderr, "can't initialize WinSocks\n");
		return;
	}
#endif

	import_array();

	/* Create the module and add the functions */
	module = Py_InitModule("dtcam", py_methods);
	if (module == NULL)
		return;

	/* Add version to the module. */
	if (PyModule_AddStringConstant(module, "__version__", MODULE_VERSION) == -1)
		return;

	/* create a new type */
    if (PyType_Ready(&PyDtcamType) < 0)
        return;

    Py_INCREF(&PyDtcamType);
    PyModule_AddObject(module, "cam", (PyObject*)&PyDtcamType);
}

