/****************************************************************************/
/**
 * \file dtcam.c
 * \brief Image capture library for DTI interferometers
 * \version 0.3
 * \author Dimitri Denk
 * \date 02.02.2011
 *
 * This file describes communication functions.
 *
 * Copyright (c) 2011 
 * DENKTECH
 * www.denktech.de
 *
 * Permission to use, copy, modify, distribute and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation. DENKTECH makes no representations about 
 * the suitability of this software for any purpose. 
 * It is provided "as is" without express or implied warranty.
 */
/****************************************************************************/

#include <stdio.h>	/* snprintf */
#include <memory.h>
#include <stdlib.h>
#include <errno.h>

#ifdef _WIN32
#include <winsock.h>
#define snprintf _snprintf
#else
#include <sys/socket.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <unistd.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <sys/ioctl.h>
/** \brief socket close definition */
#define closesocket	close
#endif

#include "dtcam.h"

/** \def TRACE(...)
 * \brief output debug information
 */
#ifdef _DEBUG
#define TRACE(...) fprintf(stdout,  __VA_ARGS__)
#else
#define TRACE(...)
#endif

/* Forward declaration */
static int extract_int(const char* buf, unsigned long size, int *val, char end);
static int decodeP5(const char* buf, unsigned long size, unsigned short* width, unsigned short* height, unsigned short *max);
static int dispatch(char* str, size_t size, int* len, unsigned char* type, char* err_buf, size_t err_size);

/* Implementation */
int perform_http_request(
	unsigned long addr,
	unsigned short port,
	const char* uri,
	unsigned char* buf,
	size_t size,
	char* err_buf,
	size_t err_size
	)
{
	int result;
	int sock;
	struct sockaddr_in haddr;
    struct timeval timeout;
	fd_set read_write_handles;
//	fd_set write_handles;
	unsigned long none_block = 1;

	snprintf(err_buf, err_size, "no error");

	/* set timeout */
	timeout.tv_sec = 1;
	timeout.tv_usec = 0;

	result = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
	if(result >= 0) {
		sock = result;
		TRACE("Socket created %d.\n", sock);

#ifdef _WIN32
		result = ioctlsocket(sock, FIONBIO, &none_block);
#else
		result = ioctl(sock, FIONBIO, &none_block);
#endif
		if(result == 0) {
			memset(&haddr, 0, sizeof(haddr));
			haddr.sin_family = AF_INET;
			haddr.sin_port = htons(port);
			memcpy(&haddr.sin_addr, &addr, 4);
			result = connect(sock, (struct sockaddr*)&haddr, sizeof(haddr));

			FD_ZERO(&read_write_handles);
			FD_SET(sock, &read_write_handles);
			timeout.tv_sec = 1;
			timeout.tv_usec = 0;
			result = select(sock + 1, NULL, &read_write_handles, NULL, &timeout);
			if(result == 1) {
				TRACE("Connected to server, port %d, result %d\n", htons(haddr.sin_port), result);
				result = 0;
			}
			else {
				snprintf(err_buf, err_size, "cannot connect: timeout (%s)", strerror(errno));
				result = -1;
			}
		}
		else {
			snprintf(err_buf, err_size, "cannot set non blocking mode: %s\n", strerror(errno));
		}

		if(result == 0) {
			int req_size = 0;
			req_size += snprintf((char*)buf, size - req_size,
				"GET %s HTTP/1.0\n\n", uri);
			TRACE("REQ:\n%s", (char*)buf);
			result = send(sock, (const char*)buf, req_size, 0);
			if(result > 0) {
				int ans_size = 0;
				do {
					/* use select() to avoid blocking on accept */
					FD_ZERO(&read_write_handles);
					FD_SET(sock, &read_write_handles);
					result = select(sock + 1, &read_write_handles, NULL, NULL, &timeout);
					TRACE("select result %d\n", result);
					if(result > 0) {
						result = recv(sock, ((char*)buf) + ans_size, size - ans_size, 0);
						TRACE("recv %d\n", result);
						if(result > 0) {
							ans_size += result;
						}
					}
					else {
						snprintf(err_buf, err_size, "receive timeout (%s)", strerror(errno));
						result = -1;
					}
				}
				while(result > 0);

				if(result == 0) {
					result = ans_size;
				}
				else {
					snprintf(err_buf, err_size, "receive error: %s", strerror(errno));
				}
			}
			else {
				snprintf(err_buf, err_size, "cannot send: %s", strerror(errno));
			}
		}
		closesocket(sock);
	}
	else {
		snprintf(err_buf, err_size, "cannot create socket: %s", strerror(errno));
	}
	TRACE("%s\n", err_buf);
	return result;
}

int process_http_response(
	unsigned char* buf,
	size_t size,
	unsigned char* type,
	char* err_buf,
	size_t err_size
	)
{
	int result;
	int len;

	snprintf(err_buf, err_size, "no error");
	
	result = dispatch((char*)buf, size, &len, type, err_buf, err_size);
	if(result > 0) {
		memcpy(buf, buf + result, size - result);
		result = size - result;
	}
	return result;
}

extern int extract_pgm(
	unsigned char* buf,
	size_t size,
	unsigned short* width,
	unsigned short* height,
	unsigned char *bpp,
	char* err_buf,
	size_t err_size
	)
{
	int result = -1;
	if((size >= 2) && (buf[0] == 'P')) {
		unsigned short max = 0;
		unsigned long imglen;
		unsigned int type;
		type = buf[1] - '0';
		switch(type) {
		case 5:
			result = decodeP5((char*)buf, size, width, height, &max);
			if(result < 0) {
				snprintf(err_buf, err_size, "error in PGM");
			}
			break;
		default:
			snprintf(err_buf, err_size, "unknown PGM type P%d", type);
			break;
		}
		if(result > 0) {
			TRACE("P5\n%d %d\n%d\n", *width, *height, max);
			imglen = *width * *height;
			if(max > 255) {
				unsigned long i;
				const unsigned char* src = buf + result;
				unsigned short* dest = (unsigned short*)buf;
				for(i = 0; i < imglen; i++) {
					*dest = *src;
					src++;
					(*dest) <<= 8;
					(*dest) |= *src;
					src++;
					dest++;
				}
				imglen *= 2;
			}
			else {
				if(imglen + result <= size)
				{
					memcpy(buf, buf + result, imglen);
					result = imglen;
				}
				else
				{
					snprintf(err_buf, err_size, "buffer too small (%ld, need %ld)",
						(unsigned long)size, (unsigned long)imglen);
				}
			}
			*bpp = 0;
			while(max > 0) {
				(*bpp)++;
				max >>= 1;
			}
		}
	}
	else {
		snprintf(err_buf, err_size, "cannot extract PGM, header not valid");
	}
	return result;
}

static int extract_int(const char* buf, unsigned long size, int *val, char end)
{
	unsigned long i = 0;
	*val = 0;
	while(i < size)
	{
		register char ch = buf[i];
		i++;
		if(ch == end)
			return i;
		else if(ch >= '0' && ch <= '9')
			*val = *val * 10 + ch - '0';
		else
			return -1;
	}
	return -1;
}

static int decodeP5(const char* buf, unsigned long size, unsigned short* width, unsigned short* height, unsigned short *max)
{
	int result = -1;
	int len = 0;
	int val;
	if(memcmp(buf, "P5\n", 3) == 0) {
		result = 3;
	}
	if(result > 0) {
		len += result;
		result = extract_int(&buf[len], size - len, &val, ' ');
		*width = val;
	}
	if(result > 0) {
		len += result;
		result = extract_int(&buf[len], size - len, &val, '\n');
		*height = val;
	}
	if(result > 0) {
		len += result;
		result = extract_int(&buf[len], size - len, &val, '\n');
		*max = val;
	}
	if(result > 0) {
		len += result;
		return len;
	}	
	return result;
}

static int dispatch(char* str, size_t size, int* len, unsigned char* type, char* err_buf, size_t err_size)
{
	int result = -1;
	int val = 0;
	int end = 0;
	size_t n = 0;
	const char* name = NULL;
	const char* value = NULL;
	*len = 0;
	*type = 0;
	
	if((size >= 14) && ((memcmp(str, "HTTP/1.0 ", 9) == 0) || (memcmp(str, "HTTP/1.1 ", 9) == 0))) {
		if((size >= 17) && (memcmp(str + 9, "200 OK\r\n", 8) == 0)) {
			n = 17;
			str += n;
			while(size > n)
			{
				switch(*str)
				{
				case 0:
					size = n;
					break;
				case ':':
					if(!val)
					{
						*str = 0;
						val = 1;
					}
					break;
				case '\n':
					*str = 0;
					if(end)
					{
						if(*len == 0)
							*len = size - n - 1;
						size = n;
						result = n + 1;
					}
					else
					{
						end = 1;
						val = 0;
						if((value) && (strcmp(name, "Content-Length") == 0))
						{
							*len = atoi(value);
						}
						else if((value) && ((strcmp(name, "Content-Type") == 0) || ((strcmp(name, "Content-type") == 0))))
						{
							if(strcmp(value, "image/png") == 0)
								*type = CONTENT_TYPE_PNG;
							else if(strcmp(value, "image/jpeg") == 0)
								*type = CONTENT_TYPE_JPEG;
							else if(strcmp(value, "image/x-portable-graymap") == 0)
								*type = CONTENT_TYPE_PGM;
						}
						name = NULL;
						value = NULL;
					}
					break;
				case '\r':
					*str = 0;
					break;
				default:
					end = 0;
					if(val)
					{
						if(!value && *str != ' ')
						{
							value = str;
						}
					}
					else
					{
						if(!name && *str != ' ')
						{
							name = str;
						}
					}
				}
				str++;
				n++;
			}
		}
		else {
			str += 9;
			value = str;
			str = (char*)memchr(str, '\r', size - 9);
			if(str) {
				*str = 0;
				snprintf(err_buf, err_size, "HTTP response: %s", value);
			}
			else {
				snprintf(err_buf, err_size, "cannot dispatch response");
			}
		}
	}
	else {
		snprintf(err_buf, err_size, "cannot dispatch response");
	}
	
	return result;
}

/* EOF */
