/****************************************************************************/
/**
 * \file dtcam.h
 * \brief Image capture library for DTI interferometers
 * \version 0.3
 * \author Dimitri Denk
 * \date 02.02.2010
 *
 * This file describes DTI communication functions.
 *
 * Copyright (c) 2010 
 * DENKTECH
 * www.denktech.de
 *
 * Permission to use, copy, modify, distribute and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation. DENKTECH makes no representations about 
 * the suitability of this software for any purpose. 
 * It is provided "as is" without express or implied warranty.
 */
/****************************************************************************/

/** \addtogroup LIB_DTCAM libdtcam
 * \ingroup DTI_TOOLS
 *
 * Image capture library for DTI interferometers
 */
/*@{*/

#ifndef DTCAMC_H
#define DTCAMC_H

#ifdef __cplusplus
extern "C" {
#endif

/** \brief  Content-Type: image/png */
#define CONTENT_TYPE_PNG	1
/** \brief Content-Type: image/jpeg */
#define CONTENT_TYPE_JPEG	2
/** \brief Content-Type: image/x-portable-graymap */
#define CONTENT_TYPE_PGM	3

/** \brief Perform HTTP request
 *
 * The perform_http_request function creates socket and sends HTTP request to
 * server, waits response and stores it in buffer.
 *
 * \param addr - (In) server IP address
 * \param port - (In) server port
 * \param uri - (In) resource name
 * \param buf - (Out) response from server
 * \param size - (In) buffer size
 * \param err_buf - (Out) buffer for error message
 * \param err_size - (In) maximal size of error message
 * \retval size - response size, <0 if error
 */
extern int perform_http_request(
	unsigned long addr,
	unsigned short port,
	const char* uri,
	unsigned char* buf,
	size_t size,
	char* err_buf,
	size_t err_size
	);

/** \brief Process HTTP response
 *
 * The process_http_response function parse response from HTTP server.
 *
 * \param buf - (In-Out) response from server
 * \param size - (In) buffer size
 * \param type - (Out) content type
 * \param err_buf - (Out) buffer for error message
 * \param err_size - (In) maximal size of error message
 * \retval size - content size, <0 if buffer data is not HTTP response or respponse code not 200
 */
extern int process_http_response(
	unsigned char* buf,
	size_t size,
	unsigned char* type,
	char* err_buf,
	size_t err_size
	);

/** \brief Extract binary array form PGM image format
 *
 * The extract_pgm function convert PGM data format to binary array.
 *
 * \param buf - (In-Out) PGM image
 * \param size - (In) buffer size
 * \param width - (Out) image width
 * \param height - (Out) image height
 * \param bpp - (Out) image digitization (bits per pixel)
 * \param err_buf - (Out) buffer for error message
 * \param err_size - (In) maximal size of error message
 * \retval size - image size, <0 if buffer data is not pgm
 * \see PGM Format Specification
 */
extern int extract_pgm(
	unsigned char* buf,
	size_t size,
	unsigned short* width,
	unsigned short* height,
	unsigned char *bpp,
	char* err_buf,
	size_t err_size
	);

#ifdef  __cplusplus
}
#endif

#endif /*DTCAMC_H*/

/*@}*/

/* EOF */
