/****************************************************************************/
/**
 * \file dtcam.c
 * \brief Matlab image capture library for DTI interferometers
 * \version 0.3
 * \author Dimitri Denk
 * \date 02.02.2011
 *
 * This file implements Mathlab integration of DTI communication functions.
 *
 * Copyright (c) 2011 
 * DENKTECH
 * www.denktech.de
 *
 * Permission to use, copy, modify, distribute and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation. DENKTECH makes no representations about 
 * the suitability of this software for any purpose. 
 * It is provided "as is" without express or implied warranty.
 */
/****************************************************************************/
#include "mex.h"

#include <stdio.h>	/* snprintf */

#include "dtcam.h"

static char err_buf[256];
unsigned long cam_ip = 0;
unsigned short cam_port = 80;
unsigned char buf[1024 * 1024 * 2];

void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
//	mexPrintf("nlhs %d, nrhs %d\n", nlhs, nrhs);

	/* Check for proper number of arguments. */
	if(nrhs < 1) {
		mexErrMsgTxt("input required.");
	}
	else {
		if((mxGetClassID(prhs[0]) != mxUINT8_CLASS) || mxIsComplex(prhs[0]) || (mxGetM(prhs[0]) != 1) || (mxGetN(prhs[0]) != 1)) {
			mexErrMsgTxt("parameter must be a noncomplex scalar uint8.");
		}
		else {
			unsigned char func;
			func = *(unsigned char*)mxGetPr(prhs[0]);
			switch(func) {
			/* set ip and port */
			case 0:
				if(nlhs > 0) {
					mexErrMsgTxt("too many output arguments");
				}
				else if(nrhs != 3) {
					mexErrMsgTxt("three parameters required.");
				}
				else if((mxGetClassID(prhs[1]) != mxUINT8_CLASS) || mxIsComplex(prhs[1]) || (mxGetM(prhs[1]) != 1) || (mxGetN(prhs[1]) != 4)) {
					mexErrMsgTxt("second parameter must be uint8 vector of size four.");
				}
				else if((mxGetClassID(prhs[2]) != mxUINT16_CLASS) || mxIsComplex(prhs[2]) || (mxGetM(prhs[2]) != 1) || (mxGetN(prhs[2]) != 1)) {
					mexErrMsgTxt("third input must be uint16.");
				}
				else {
					unsigned char* addr;
					unsigned short* port;
					addr = (unsigned char*)mxGetPr(prhs[1]);
					port = (unsigned short*)mxGetPr(prhs[2]);
					if((addr) && (port)) {
//						mexPrintf("address %d.%d.%d.%d:%d\n", addr[0], addr[1], addr[2], addr[3], *port);
						cam_ip = addr[3];
						cam_ip = (cam_ip << 8) + addr[2];
						cam_ip = (cam_ip << 8) + addr[1];
						cam_ip = (cam_ip << 8) + addr[0];
						cam_port = *port;
					}
				}
				break;
			/* get image */
			case 1:
				if(nlhs != 1) {
					mexErrMsgTxt("one output argument required.");
				}
				else if(nrhs > 2) {
					mexErrMsgTxt("too many parameters");
				}
				else if((nrhs == 2) && ((mxGetClassID(prhs[1]) != mxINT16_CLASS) || mxIsComplex(prhs[1]) || (mxGetM(prhs[1]) != 1) || (mxGetN(prhs[1]) != 1))) {
					mexErrMsgTxt("second parameter must be int16.");
				}
				else {
					char uri[256];
					int result;
					unsigned char type;
					unsigned short width, height;
					unsigned char bpp;
					if(nrhs == 2) {
						short* shift;
						shift = (short*)mxGetPr(prhs[1]);
						snprintf(uri, sizeof(uri), "/img.pgm?shift=%d", shift[0]); 
					}
					else {
						snprintf(uri, sizeof(uri), "/img.pgm"); 
					}
//					mexPrintf("uri \"%s\"\n", uri);
					result = perform_http_request(
						cam_ip, cam_port, uri,
						buf, sizeof(buf),
						err_buf, sizeof(err_buf)
						);
//						mexPrintf("perform_http_request %d\n", result);
					if(result >= 0) {
						result = process_http_response(
							buf, result,
							&type,
							err_buf, sizeof(err_buf)
						);
//						mexPrintf("process_http_response %d\n", result);
					}
					if(result >= 0) {
						if(type == 3) {
							result = extract_pgm(
								buf, result,
								&width, &height, &bpp,
								err_buf, sizeof(err_buf)
								);
//								mexPrintf("extract_pgm %d\n", result);
							if(result >= 0) {
								unsigned short i, j;
								unsigned short* dest;
//								mexPrintf("image %dbpp %dx%d\n", bpp, height, width);
								plhs[0] = mxCreateNumericMatrix(height, width, mxUINT16_CLASS, mxREAL);
								if(bpp > 8) {
									const unsigned short *src;
									src = (const unsigned short*)buf;
									for(j = 0; j < height; j++) {
										dest = (unsigned short*)mxGetPr(plhs[0]) + j;
										for(i = 0; i < width; i++) {
											*dest = (*src) << (16 - bpp);
											src++;
											dest += height;
										}
									}
								}
								else {
									const unsigned char *src;
									src = (const unsigned char*)buf;
									for(j = 0; j < height; j++) {
										dest = (unsigned short*)mxGetPr(plhs[0]) + j;
										for(i = 0; i < width; i++) {
											*dest = ((unsigned short)(*src)) << (16 - bpp);
											src++;
											dest += height;
										}
									}
								}
							}
						}
						else {
							mexErrMsgTxt("wrong content type");
						}
					}
					if(result < 0) {
						mexErrMsgTxt(err_buf);
					}
				}
				break;
			/* get phase */
			case 2:
				if(nlhs != 1) {
					mexErrMsgTxt("one output argument required.");
				}
				else if(nrhs != 2) {
					mexErrMsgTxt("two parameters required.");
				}
				else if((mxGetClassID(prhs[1]) != mxINT16_CLASS) || mxIsComplex(prhs[1]) || (mxGetM(prhs[1]) != 1) || (mxGetN(prhs[1]) != 5)) {
					mexErrMsgTxt("parameter must be int16 vector of size 5.");
				}
				else {
					short* shift;
					char uri[256];
					shift = (short*)mxGetPr(prhs[1]);
					if(shift) {
						int result;
						unsigned char type;
						unsigned short width, height;
						unsigned char bpp;
						snprintf(uri, sizeof(uri),
							"/phase.pgm?shift=%d&shift=%d&shift=%d&shift=%d&shift=%d",
							shift[0], shift[1], shift[2], shift[3], shift[4]
							); 
//						mexPrintf("uri \"%s\"\n", uri);
						result = perform_http_request(
							cam_ip, cam_port, uri,
							buf, sizeof(buf),
							err_buf, sizeof(err_buf)
							);
//						mexPrintf("perform_http_request %d\n", result);
						if(result >= 0) {
							result = process_http_response(
								buf, result,
								&type,
								err_buf, sizeof(err_buf)
							);
//							mexPrintf("process_http_response %d\n", result);
//							mexPrintf("error %s\n", err_buf);
						}
						if(result >= 0) {
							if(type == 3) {
								result = extract_pgm(
									buf, result,
									&width, &height, &bpp,
									err_buf, sizeof(err_buf)
									);
//								mexPrintf("extract_pgm %d\n", result);
								if(result >= 0) {
									unsigned short i, j;
//									mexPrintf("image %dbpp %dx%d\n", bpp, height, width);
									plhs[0] = mxCreateNumericMatrix(height, width, mxUINT16_CLASS, mxREAL);
									if(bpp == 16) {
										const unsigned short *src = (unsigned short*)buf;
										unsigned short* dest;
										for(j = 0; j < height; j++) {
											dest = (unsigned short*)mxGetPr(plhs[0]) + j;
											for(i = 0; i < width; i++) {
												*dest = (*src);
												src++;
												dest += height;
											}
										}
									}
									else {
										mexErrMsgTxt("wrong pgm resolution");
									}
								}
							}
							else {
								mexErrMsgTxt("wrong content type");
							}
						}
						if(result < 0) {
							mexErrMsgTxt(err_buf);
						}
					}
				}
				break;
			/* unknown function */
			default:
				mexErrMsgTxt("unknown function");
				break;
			}
		}
	}
}
